<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-10-18
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer;

use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\CustomerFactory as CustomerEntityFactory;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer\Address as AddressHydrationWorker;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Newsletter\Model\Subscriber;

/**
 * Class Data
 *
 * Hydration class used for inserting or updating customers from Magento to Fastmag
 */
class Data extends Customer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_customer';

    /** @var Subscriber $subscriber */
    protected $subscriber;

    /** @var AddressHydrationWorker $addressHydrationWorker */
    protected $addressHydrationWorker;

    /** @inheritDoc */
    protected $subordinateWorkersAfter = ['tofastmag_hydration_customer_address'];

    /**
     * Customer constructor.
     *
     * @param Logger                   $logger
     * @param JobRepository            $jobRepository
     * @param Config                   $config
     * @param Json                     $jsonSerializer
     * @param SyncedCustomerRepository $syncedCustomerRepository
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param CustomerRepository       $customerRepository
     * @param CustomerEntityFactory    $customerEntityFactory
     * @param Subscriber               $subscriber
     * @param AddressHydrationWorker   $addressHydrationWorker
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        SyncedCustomerRepository $syncedCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CustomerRepository $customerRepository,
        CustomerEntityFactory $customerEntityFactory,
        Subscriber $subscriber,
        AddressHydrationWorker $addressHydrationWorker
    ) {
        parent::__construct(
            $logger,
            $jobRepository,
            $config,
            $jsonSerializer,
            $syncedCustomerRepository,
            $searchCriteriaBuilder,
            $customerRepository,
            $customerEntityFactory
        );

        $this->subscriber = $subscriber;
        $this->addressHydrationWorker = $addressHydrationWorker;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        $entities = false;

        try {
            $entities = $this->getDataFromMagento();
        } catch (ProcessException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }

        if (is_array($entities)) {
            foreach ($entities as $customerId => $entity) {
                $job = $this->getJob($customerId);

                if ($job !== null) {
                    $this->hydrateJob($job, $entity);
                }
            }
        }

        foreach ($this->getJobs()->getItems() as $job) {
            $this->saveJob($job);
        }
    }

    /**
     * @inheritDoc
     *
     * @return CustomerEntity[]
     *
     * @throws ProcessException
     * @throws JobException
     */
    protected function getDataFromMagento()
    {
        $this->entities = [];
        $customersIds = $this->getJobs()->getColumnValues(Job::CONTENT_ID);

        foreach ($customersIds as $customerId) {
            $customerEntity = $this->customerEntityFactory->create();
            $customerEntity->setMagentoId($customerId);
            $this->entities[$customerId] = $customerEntity;
        }

        $this->getSyncedCustomers($customersIds);
        $this->getMagentoCustomersData($customersIds);

        return $this->entities;
    }
}
