<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-23
 ******************************************************************************/

namespace Fastmag\Sync\Model\System\Connection\Sql;

use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config\Provider\Edi as EdiConfig;
use Magento\Framework\HTTP\Client\Curl;

/**
 * Class EdiSql
 *
 * Class to connect and query to Fastmag through EDI
 */
class EdiSql implements SqlInterface
{
    /** @var EdiConfig $ediConfigProvider */
    protected $ediConfigProvider;

    /** @var Curl $curlClient */
    protected $curlClient;

    /** @var Logger $logger */
    protected $logger;

    /** @var array $connectionConfig */
    protected $connectionConfig;

    /** @var string $endpoint */
    protected $endpoint = 'ediquery.ips';

    /**
     * Edi constructor
     *
     * @param EdiConfig $ediConfigProvider
     * @param Curl      $curlClient
     * @param Logger    $logger
     */
    public function __construct(EdiConfig $ediConfigProvider, Curl $curlClient, Logger $logger)
    {
        $this->ediConfigProvider = $ediConfigProvider;
        $this->curlClient = $curlClient;
        $this->logger = $logger;

        $this->connectionConfig = $this->getCredentialsFromConfig();
    }

    /**
     * Try connect to EDI
     *
     * @return bool
     *
     * @throws NoConnectionException
     *
     * @todo find a better way to test the connection
     */
    public function connect()
    {
        $this->get('SELECT NOW()');
        $response = $this->curlClient->getBody();

        if (strpos($response, 'KO|') !== false) {
            throw new NoConnectionException(__($response));
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function get($sql)
    {
        $result = [];

        $curlOptions = [
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_RETURNTRANSFER => true
        ];

        $this->postCurl(trim($sql), $curlOptions);
        $response = $this->getResponseBodyCurl();

        if ($response !== false) {
            $lines = explode("\n", $response);

            if (strpos($response, 'KO|') !== false) {
                if (strpos($response, 'KO|ERREUR') !== false) {
                    throw new NoConnectionException(__($response));
                }
            } elseif (strpos($response, 'Your EDI account has been blocked') !== false) {
                throw new NoConnectionException(__($response));
            } else {
                foreach ($lines as $line) {
                    $line = str_replace(['<CRLF>', "\r"], ["\n", ''], $line);

                    if (trim($line) !== '') {
                        $row = explode("\t", $line);
                        $result[] = $row;
                    }
                }

                array_walk($result, static function (&$array) use ($result) {
                    $array = @array_combine($result[0], $array);
                });

                // Remove column line
                array_shift($result);
            }
        }

        return $result;
    }

    /**
     * Post data to EDI
     *
     * @param string $ediData
     * @param array  $params
     *
     * @return array
     *
     * @throws NoConnectionException
     */
    public function post($sql, $params = [])
    {
        $result = $this->get($sql);

        return is_array($result);
    }

    /**
     * @inheritDoc
     *
     * @todo
     */
    public function multiPost($sql)
    {
        return $this->post($sql);
    }

    /**
     * @inheritDoc
     */
    public function escape($value, $forceString = false)
    {
        if (is_array($value)) {
            $result = '';

            foreach ($value as $row) {
                $result .= $this->escape($row, $forceString) . ', ';
            }

            $result = trim($result, ', ');
        } else {
            $result = stripslashes($value);

            if ($forceString || !is_numeric($result)) {
                $search = ['\\', "\0", "\n", "\r", "\x1a", "'", '"'];
                $replace = ['\\\\', '\\0', '\\n', '\\r', "\Z", "\'", '\"'];

                $result = '\'' . strip_tags(nl2br(str_replace($search, $replace, $result))) . '\'';
            }
        }

        return $result;
    }

    /**
     * Set API access credentials
     *
     * @param string $host
     * @param string $port
     * @param string $chain
     * @param string $shop
     * @param string $user
     * @param string $password
     * @param bool   $ssl
     *
     * @return $this
     */
    public function setCredentials($host, $port, $chain, $shop, $user, $password, $ssl)
    {
        $this->connectionConfig = [
            'url'      => $this->getUrl($host, $port, $ssl),
            'chain'    => $chain,
            'shop'     => $shop,
            'username' => $user,
            'password' => $password
        ];

        return $this;
    }

    /**
     * Returns data source (given by config field)
     *
     * @return string
     */
    public function getDataSource()
    {
        return 'fastmag_' . $this->getChain();
    }

    /**
     * Returns chain
     *
     * @return string
     */
    public function getChain()
    {
        return $this->connectionConfig['chain'];
    }

    /**
     * Get Direct SQL config stored in conf
     *
     * @return array
     */
    protected function getCredentialsFromConfig()
    {
        return [
            'url'      => $this->getUrl(
                $this->ediConfigProvider->getHost(),
                $this->ediConfigProvider->getPort(),
                $this->ediConfigProvider->getSsl()
            ),
            'chain'    => $this->ediConfigProvider->getChain(),
            'shop'     => $this->ediConfigProvider->getShop(),
            'username' => $this->ediConfigProvider->getUser(),
            'password' => $this->ediConfigProvider->getPassword()
        ];
    }

    /**
     * Get EDI URL
     *
     * @param string $host
     * @param string $port
     * @param bool   $ssl
     *
     * @return string|null
     */
    protected function getUrl($host, $port, $ssl = true)
    {
        return 'http' . ($ssl ? 's' : '') . '://' . $host . ($port ?: '') . '/' . $this->endpoint;
    }

    /**
     * Internal method to send data to EDI through CURL
     *
     * @param string $sql
     * @param array  $curlOptions
     *
     * @return float
     */
    protected function postCurl($sql, $curlOptions = [])
    {
        $start = microtime(true);

        $this->curlClient->setOptions($curlOptions);

        $this->curlClient->post(
            $this->connectionConfig['url'],
            [
                'compte'   => $this->connectionConfig['username'],
                'motpasse' => $this->connectionConfig['password'],
                'enseigne' => $this->connectionConfig['chain'],
                'magasin'  => $this->connectionConfig['shop'],
                'data'     => $sql
            ]
        );

        $end = microtime(true);

        return $end - $start;
    }

    /**
     * Returns last response body
     *
     * @return string
     */
    protected function getResponseBodyCurl()
    {
        return $this->curlClient->getBody();
    }

    /**
     * Returns last response headers
     *
     * @return array
     */
    protected function getResponseHeadersCurl()
    {
        return $this->curlClient->getHeaders();
    }

    /**
     * Returns last response status
     *
     * @return int
     */
    protected function getResponseStatusCurl()
    {
        return $this->curlClient->getStatus();
    }
}
