<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-10-29
 ******************************************************************************/

namespace Fastmag\Sync\Process\Entity\ToFastmag;

use Fastmag\Sync\Api\Data\OrderInterface as SyncOrder;
use Fastmag\Sync\Process\EntityInterface;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\Item as ItemEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\ItemFactory as ItemEntityFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\Transfer as TransferEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\TransferFactory as TransferEntityFactory;
use Magento\Framework\DataObject;
use Magento\Sales\Model\Order\Item;

/**
 * Class Order
 *
 * Entity class for order.
 * Save and transform data from Magento for Fastmag easy integration,
 * and transfer it from hydration worker to integration worker
 *
 * @method int|null getMagentoId()
 * @method int|null getIncrementId()
 * @method string|null getStatus()
 * @method int|null getTotalQty()
 * @method float|null getGrandTotal()
 * @method ItemEntity[]|null getItems()
 * @method float|null getShippingRate()
 * @method float|null getShippingDiscount()
 * @method string|null getShippingDescription()
 * @method string|null getPaymentId()
 * @method string|null getPaymentCode()
 * @method string|null getTransactionType()
 * @method string|null getCarrierCode()
 * @method SyncOrder|null getLastTransaction()
 * @method int|null getCustomerId()
 * @method Customer|null getCustomer()
 * @method Address|null getBillingAddress()
 * @method Address|null getShippingAddress()
 * @method TransferEntity[]|null getTransfers()
 * @method string getFastmagChain()
 * @method string getFastmagShop()
 * @method string getFastmagSeller()
 * @method string getFastmagReferenceStock()
 * @method string getSendStatus()
 * @method bool getIsPassedAsGuest()
 * @method string[] getFastmagStocks()
 * @method Order setMagentoId(int $magentoId)
 * @method Order setIncrementId(int $incrementId)
 * @method Order setStatus(string $status)
 * @method Order setTotalQty(int $totalQty)
 * @method Order setIsExcludedTax(bool $isExcludedTax)
 * @method Order setGrandTotal(float $grandTotal)
 * @method Order setItems(ItemEntity[] $items)
 * @method Order setShippingRate(float $shippingRate)
 * @method Order setShippingDiscount(float $shippingDiscount)
 * @method Order setShippingDescription(float $shippingDescription)
 * @method Order setPaymentId(string $paymentId)
 * @method Order setPaymentCode(string $paymentCode)
 * @method Order setTransactionType(string $transactionType)
 * @method Order setCarrierCode(string $carrierCode)
 * @method Order setLastTransaction(SyncOrder $lastTransaction)
 * @method Order setCustomerId(int $customerId)
 * @method Order setCustomer(Customer $customer)
 * @method Order setBillingAddress(Address $address)
 * @method Order setShippingAddress(Address $address)
 * @method Order setTransfers(TransferEntity[] $transfers)
 * @method Order setFastmagChain(string $chain)
 * @method Order setFastmagShop(string $shop)
 * @method Order setFastmagSeller(string $seller)
 * @method Order setFastmagReferenceStock(string $stocks)
 * @method Order setSendStatus(bool $sendStatus)
 * @method Order setIsPassedAsGuest(bool $isPassedAsGuest)
 */
class Order extends DataObject implements EntityInterface
{
    /** @var ItemEntityFactory $itemEntityFactory */
    protected $itemEntityFactory;

    /** @var TransferEntityFactory $transferEntityFactory */
    protected $transferEntityFactory;

    /**
     * Order constructor.
     *
     * @param ItemEntityFactory     $itemEntityFactory
     * @param TransferEntityFactory $transferEntityFactory
     * @param array                 $data
     */
    public function __construct(
        ItemEntityFactory $itemEntityFactory,
        TransferEntityFactory $transferEntityFactory,
        array $data = []
    ) {
        parent::__construct($data);

        $this->itemEntityFactory = $itemEntityFactory;
        $this->transferEntityFactory = $transferEntityFactory;
    }

    /**
     * Is order excluded tax
     *
     * @return bool|null
     */
    public function isExcludedTax()
    {
        return $this->getData('is_excluded_tax') !== null ? (bool)$this->getData('is_excluded_tax') : null;
    }

    /**
     * Get item by Fastmag ID
     *
     * @param int $fastmagId
     *
     * @return ItemEntity|null
     */
    public function getItemByFastmagId($fastmagId)
    {
        $result = null;

        foreach ($this->getItems() as $item) {
            if ($item->getFastmagId() === $fastmagId) {
                $result = $item;

                break;
            }
        }

        return $result;
    }

    /**
     * Get item by Magento ID
     *
     * @param int $magentoId
     *
     * @return ItemEntity|null
     */
    public function getItemByMagentoId($magentoId)
    {
        $result = null;

        if (array_key_exists($magentoId, $this->getItems())) {
            $result = $this->getItems()[$magentoId];
        }

        return $result;
    }

    /**
     * Returns alternative stocks as an array, even if when there's no alternative stock, or when there's only one
     */
    public function getFastmagAlternativeStocks()
    {
        $result = $this->getData('fastmag_alternative_stocks');

        if (count($result) === 0) {
            $result = [];
        }

        return $result;
    }

    /**
     * Returns the list of all reference and alternatives stocks, deduplicated and with reference stock first
     *
     * @return string[]
     */
    public function getAllFastmagStocks()
    {
        return array_unique(
            array_merge([$this->getData('fastmag_reference_stock')], $this->getFastmagAlternativeStocks())
        );
    }

    /**
     * Add item to the items list
     *
     * @param Item $item
     *
     * @return Order
     */
    public function addItem($item)
    {
        $items = $this->getData('items');

        $itemEntity = $this->itemEntityFactory->create();

        $itemEntity->setMagentoId($item->getItemId())
            ->setParentItemId($item->getParentItemId())
            ->setFastmagId($item->getFastmagProductId())
            ->setName($item->getName())
            ->setPriceInclTax($item->getBasePriceInclTax())
            ->setQtyOrdered($item->getQtyOrdered())
            ->setQtyCanceled($item->getQtyCanceled())
            ->setDiscountAmount($item->getBaseDiscountAmount())
            ->setRowTotalInclTax($item->getBaseRowTotalInclTax());

        if ($item->getParentItemId()) {
            $parentItem = $item->getParentItem();

            if ($parentItem !== null) {
                $itemEntity->setPriceInclTax($parentItem->getBasePriceInclTax())
                    ->setDiscountAmount($parentItem->getBaseDiscountAmount())
                    ->setRowTotalInclTax($parentItem->getBaseRowTotalInclTax());
            }
        }

        $items[$itemEntity->getMagentoId()] = $itemEntity;

        return $this->setData('items', $items);
    }

    /**
     * Set Fastmag alternative stocks
     *
     * @param string $stocks
     *
     * @return Order
     */
    public function setFastmagAlternativeStocks($stocks)
    {
        if (strpos($stocks, '|') !== false) {
            $stocksList = explode('|', $stocks);

            $this->setData('fastmag_alternative_stocks', $stocksList);
        } else {
            $this->setData('fastmag_alternative_stocks', $stocks);
        }

        return $this;
    }

    /**
     * Add Fastmag alternative stocks to the list
     *
     * @param string $newStock
     *
     * @return Order
     */
    public function addFastmagAlternativeStocks($newStock)
    {
        if (strpos($newStock, '|') !== false) {
            $newStocksList = explode('|', $newStock);

            foreach ($newStocksList as $stock) {
                $this->addFastmagAlternativeStocks($stock);
            }
        } else {
            $stocks = $this->getData('fastmag_stocks');

            if (!in_array($newStock, $stocks, true)) {
                $stocks[] = $newStock;

                $this->setData('fastmag_stocks', $stocks);
            }
        }

        return $this;
    }

    /**
     * Add transfer to the transfers list
     *
     * @param SyncOrder $transfer
     *
     * @return Order
     */
    public function addTransfer($transfer)
    {
        $transfers = $this->getData('transfers');

        $transferEntity = $this->transferEntityFactory->create();

        $transferEntity->setSyncId($transfer->getId())
            ->setTransactionId($transfer->getTransactionId())
            ->setRequest($transfer->getRequest())
            ->setStatus($transfer->getStatus())
            ->setOriginShop($transfer->getOriginShop())
            ->setTargetShop($transfer->getTargetShop());

        $transfers[$transferEntity->getSyncId()] = $transferEntity;

        return $this->setData('transfers', $transfers);
    }

    /**
     * @inheritDoc
     */
    public function export()
    {
        $itemsData = [];
        if ($this->getItems() !== null && count($this->getItems()) > 0) {
            foreach ($this->getItems() as $item) {
                $itemsData[$item->getMagentoId()] = $item->export();
            }
        }

        $transfersData = [];
        if ($this->getTransfers() !== null && count($this->getTransfers()) > 0) {
            foreach ($this->getTransfers() as $transfer) {
                $transfersData[$transfer->getSyncId()] = $transfer->export();
            }
        }

        $result = [
            'order'   => [
                'order_id'             => $this->getMagentoId(),
                'increment_id'         => $this->getIncrementId(),
                'status'               => $this->getStatus(),
                'grand_total'          => $this->getGrandTotal(),
                'total_qty'            => $this->getTotalQty(),
                'is_excluding_tax'     => $this->isExcludedTax(),
                'shipping_rate'        => $this->getShippingRate(),
                'shipping_discount'    => $this->getShippingDiscount(),
                'shipping_description' => $this->getShippingDescription(),
                'payment_id'           => $this->getPaymentId(),
                'payment_code'         => $this->getPaymentCode(),
                'items'                => $itemsData,
                'customer'             => ($this->getCustomer() ? $this->getCustomer()->export() : null),
                'addresses'            => [
                    'billing'  => ($this->getBillingAddress() ? $this->getBillingAddress()->export() : null),
                    'shipping' => ($this->getShippingAddress() ? $this->getShippingAddress()->export() : null)
                ]
            ],
            'fastmag' => [
                'chain'                 => $this->getFastmagChain(),
                'fastmag_shop'          => $this->getFastmagShop(),
                'seller'                => $this->getFastmagSeller(),
                'reference_stock'       => $this->getFastmagReferenceStock(),
                'alternative_stocks'    => implode(', ', $this->getFastmagAlternativeStocks()),
                'send_status'           => $this->getSendStatus(),
                'transaction_type'      => $this->getTransactionType(),
                'carrier_code'          => $this->getCarrierCode(),
                'last_transaction_id'   =>
                    ($this->getLastTransaction() ? $this->getLastTransaction()->getTransactionId() : null),
                'last_transaction_type' =>
                    ($this->getLastTransaction() ? $this->getLastTransaction()->getTransactionType() : null),
                'transfers'             => $transfersData
            ]
        ];

        return array_filter(
            $result,
            static function ($value) {
                return $value !== null && $value !== [];
            }
        );
    }
}
