<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-11-03
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration;

use Exception;
use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Data\CustomerInterface as SyncedCustomerInterface;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\CustomerFactory as CustomerEntityFactory;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration as HydrationTrait;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Newsletter\Model\Subscriber;

/**
 * Class Customer
 *
 * Hydration class used for inserting or updating customers from Magento to Fastmag
 */
abstract class Customer extends Worker
{
    use HydrationTrait;

    /** @var SyncedCustomerRepository $syncedCustomerRepository */
    protected $syncedCustomerRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var CustomerRepository $customerRepository */
    protected $customerRepository;

    /** @var CustomerEntityFactory $customerEntityFactory */
    protected $customerEntityFactory;

    /** @var Subscriber $subscriber */
    protected $subscriber;

    /** @var CustomerEntity[] $entities */
    protected $entities;

    /**
     * Customer constructor.
     *
     * @param Logger                   $logger
     * @param JobRepository            $jobRepository
     * @param Config                   $config
     * @param Json                     $jsonSerializer
     * @param SyncedCustomerRepository $syncedCustomerRepository
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param CustomerRepository       $customerRepository
     * @param CustomerEntityFactory    $customerEntityFactory
     * @param Subscriber               $subscriber
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        SyncedCustomerRepository $syncedCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CustomerRepository $customerRepository,
        CustomerEntityFactory $customerEntityFactory,
        Subscriber $subscriber
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->config = $config;
        $this->jsonSerializer = $jsonSerializer;
        $this->syncedCustomerRepository = $syncedCustomerRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->customerRepository = $customerRepository;
        $this->customerEntityFactory = $customerEntityFactory;
        $this->subscriber = $subscriber;
    }

    /**
     * Get synced Fastmag ids for the current jobs
     *
     * @param int[] $customerIds
     *
     * @return void
     *
     * @throws JobException
     */
    protected function getSyncedCustomers($customerIds)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(SyncedCustomerInterface::MAGENTO_CUSTOMER_ID, $customerIds, 'in')
            ->create();

        $syncedCustomerList = $this->syncedCustomerRepository->getList($searchCriteria);

        foreach ($syncedCustomerList->getItems() as $syncedCustomer) {
            $job = $this->getJob($syncedCustomer->getMagentoCustomerId());

            if ($job === null) {
                throw new JobException(
                    __('There\'s no job for customer ID #%1', $syncedCustomer->getMagentoCustomerId())
                );
            }

            $this->log($job, 'Customer already synced, Fastmag ID: ' . $syncedCustomer->getFastmagCustomerId());

            $currentEntity = $this->entities[$syncedCustomer->getMagentoCustomerId()];
            $currentEntity->setFastmagId($syncedCustomer->getFastmagCustomerId());
        }
    }

    /**
     * Get Magento data for the current jobs
     *
     * @param int[] $customersIds
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getMagentoCustomersData($customersIds)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('entity_id', $customersIds, 'in')
            ->create();

        try {
            $customersList = $this->customerRepository->getList($searchCriteria);

            foreach ($customersList->getItems() as $customer) {
                $currentEntity = $this->getCustomerEntity($customer->getId());

                $currentEntity->setGender($customer->getGender())
                    ->setPrefix($customer->getPrefix())
                    ->setLastname($customer->getLastname())
                    ->setFirstname($customer->getFirstname())
                    ->setEmailAddress($customer->getEmail())
                    ->setStoreId($customer->getStoreId())
                    ->setDob($customer->getDob());

                $checkSubscriber = $this->subscriber->loadByCustomerId($customer->getId());
                $currentEntity->setSubscribeNewsletter($checkSubscriber->isSubscribed());
            }
        } catch (Exception $exception) {
            throw new ProcessException(__(
                'Error when hydrating customers. Message: %1. Customers IDs: %2',
                $exception->getMessage(),
                implode(', ', $customersIds)
            ));
        }
    }

    /**
     * Returns the job for the current customer ID hydrated
     *
     * @param int $customerId
     *
     * @return Job
     */
    protected function getJob($customerId)
    {
        return $this->getJobs()->getItemByColumnValue(Job::CONTENT_ID, $customerId);
    }

    /**
     * Get entity linked with customer ID
     *
     * @param int $customerId
     *
     * @return CustomerEntity
     */
    protected function getCustomerEntity($customerId)
    {
        return $this->entities[$customerId];
    }
}
