<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-10-28
 ******************************************************************************/

namespace Fastmag\Sync\Process\Manager;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Jobqueue as AbstractJob;
use Fastmag\Sync\Model\Jobqueue\ToMagentoRepository as JobRepository;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToMagento\Collection as JobCollection;
use Fastmag\Sync\Process\Manager;
use Fastmag\Sync\Process\Worker\ToMagento\Clean;
use Fastmag\Sync\Process\Worker\ToMagento\Integration;
use Fastmag\Sync\Process\Worker\ToMagento\Integration\Product\Inventory;
use Fastmag\Sync\Process\Worker\ToMagento\Scheduled;
use Fastmag\Sync\Process\WorkerFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\Exception\LocalizedException;

/**
 * Class ToMagento
 *
 * Workers manager for synchronization from Fastmag to Magento
 */
class ToMagento extends Manager
{
    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var SortOrderBuilder $sortOrderBuilder */
    protected $sortOrderBuilder;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /**
     * ToMagento constructor
     *
     * @param Logger                $logger
     * @param Config                $config
     * @param WorkerFactory         $workerFactory
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param SortOrderBuilder      $sortOrderBuilder
     * @param JobRepository         $jobRepository
     */
    public function __construct(
        Logger $logger,
        Config $config,
        WorkerFactory $workerFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger, $config, $workerFactory);

        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @return JobRepository
     */
    protected function getJobRepository()
    {
        return $this->jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @return JobCollection
     */
    protected function getCurrentJobsCollection()
    {
        return $this->currentJobsCollection;
    }

    /**
     * Main method
     *
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return void
     *
     * @throws ProcessException
     */
    public function run($jobCode = null, $jobId = null)
    {
        try {
            $checkOk = $this->checkPrerequisites($jobCode, $jobId);

            if ($checkOk) {
                $this->logger->debug('To Magento Sync - Begin process');

                $this->moveScheduledJobs();

                $jobsCollectionsByPriorityCode = $this->getJobsCollections($jobCode, $jobId);

                if (count($jobsCollectionsByPriorityCode) > 0) {
                    foreach ($jobsCollectionsByPriorityCode as $collectionPriority => $jobsCollectionByPriorityCode) {
                        foreach ($jobsCollectionByPriorityCode as $collectionJobCode => $jobsCollection) {
                            $this->processJobs($collectionJobCode, $jobsCollection);
                        }
                    }
                }

                /** @var Clean cleanWorker */
                $cleanWorker = $this->workerFactory->create('tomagento_clean');
                $cleanWorker->run();
            }
        } catch (ProcessException $exception) {
            $this->logger->critical($exception->getMessage());

            throw $exception;
        } catch (LocalizedException $exception) {
            $this->logger->critical($exception->getMessage());

            throw new ProcessException(__($exception->getMessage()));
        }
    }

    /**
     * Check required config fields to run the synchronization
     *
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return bool
     *
     * @throws ProcessException
     */
    protected function checkPrerequisites($jobCode = null, $jobId = null)
    {
        $result = parent::checkPrerequisites();

        if ($result) {
            if ($jobId !== null) {
                try {
                    $job = $this->getJobRepository()->getById($jobId);

                    $jobCode = $job->getJobCode();
                } catch (Exception $exception) {
                    throw new ProcessException(__('Job ID #%1 does not exist in queue.', $jobId));
                }
            }

            if ($jobCode !== null) {
                try {
                    $this->workerFactory->create($jobCode);
                } catch (Exception $exception) {
                    throw new ProcessException(__('There\'s no worker with code "%1".', $jobCode));
                }
            }
        }

        return $result;
    }

    /**
     * Run the specific worker to move scheduled jobs into instant queue
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function moveScheduledJobs()
    {
        if ($this->config->isSetFlag(Config::XML_PATH_PRICE_IMPORT_SPECIAL_PRICES)) {
            /** @var Scheduled $scheduledWorker */
            $scheduledWorker = $this->workerFactory->create('tomagento_move_scheduled');
            $scheduledWorker->run();
        }
    }

    /**
     * Get array of jobs collection to process
     *
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return array
     */
    protected function getJobsCollections($jobCode, $jobId)
    {
        $this->searchCriteriaBuilder->addFilter(Job::STATUS, AbstractJob::STATUS_PENDING);

        if ($jobId) {
            $this->searchCriteriaBuilder->addFilter(Job::JOB_ID, $jobId);
        } elseif ($jobCode) {
            $this->searchCriteriaBuilder->addFilter(Job::JOB_CODE, $jobCode);
        }

        $searchCriteria = $this->searchCriteriaBuilder->create();

        return $this->getJobRepository()->getListByPriorityAndCode($searchCriteria, $this->getSyncLimit());
    }

    /**
     * Return worker instance for the current job code
     *
     * @param string $jobCode
     *
     * @return Integration
     */
    protected function getJobsWorker($jobCode)
    {
        if (strpos($jobCode, 'tomagento_integration_product_') === 0
            && $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_INVENTORY)
        ) {
            $this->logger->debug(
                'Jobs ' . $jobCode . ' - "Stock only" config field set, '
                . 'switched to tomagento_integration_product_quantity worker'
            );

            /** @var Inventory $result */
            $result = $this->workerFactory->create('tomagento_integration_product_quantity');
        } else {
            $result = parent::getJobsWorker($jobCode);
        }

        return $result;
    }
}
