<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-11-18
 ******************************************************************************/

namespace Fastmag\Sync\Process\Entity\ToFastmag;

use DateTime;
use Exception;
use Fastmag\Sync\Process\Entity\Date as DateTrait;
use Fastmag\Sync\Process\EntityInterface;
use Fastmag\Sync\Process\Entity\Helper\Attribute\Gender;
use Fastmag\Sync\Process\Entity\Helper\Attribute\Prefix;
use IntlDateFormatter;
use Magento\Framework\DataObject;

/**
 * Class Customer
 *
 * Entity class for customer.
 * Save and transform data from Magento for Fastmag easy integration,
 * and transfer it from hydration worker to integration worker
 *
 * @method int|null getMagentoId()
 * @method int|null getFastmagId()
 * @method int|null getStoreId()
 * @method string|null getFirstname()
 * @method string|null getLastname()
 * @method Address|null getBillingAddress()
 * @method Address|null getShippingAddress()
 * @method Customer setMagentoId(int $magentoId)
 * @method Customer setFastmagId(int $fastmagId)
 * @method Customer setStoreId(int $storeId)
 * @method Customer setGender(int $gender)
 * @method Customer setPrefix(string $prefix)
 * @method Customer setFirstname(string $firstname)
 * @method Customer setLastname(string $lastname)
 * @method Customer setEmailAddress(string $emailAddress)
 * @method Customer setSubscribeNewsletter(bool $subscribeNewsletter)
 * @method Customer setBillingAddress(Address $address)
 * @method Customer setShippingAddress(Address $address)
 */
class Customer extends DataObject implements EntityInterface
{
    use DateTrait;

    /**
     * Customer constructor.
     *
     * @param array $data
     */
    public function __construct(array $data = [])
    {
        parent::__construct($data);

        $this->timeZoneInput = 'UTC';
        $this->timeZoneOutput = 'Europe/Paris';

        $this->configureDateFormatter();
    }

    /**
     * Get gender
     *
     * @return string|null
     */
    public function getGender()
    {
        return $this->getData('gender') !== null ? Gender::getFastmagValue($this->getData('gender')) : null;
    }

    /**
     * Get prefix
     *
     * @return string
     */
    public function getPrefix()
    {
        return $this->getData('prefix') !== null ?
            Prefix::getFastmagValue($this->getData('prefix')) : Prefix::getDefaultFastmagValue();
    }

    /**
     * Get email address
     *
     * @return string
     */
    public function getEmailAddress()
    {
        return $this->getData('email_address') !== null ? strtolower($this->getData('email_address')) : null;
    }

    /**
     * Get date of birth
     *
     * @return string|null
     */
    public function getDob()
    {
        return $this->getFormattedDate('dob', 'dd/MM/y');
    }

    /**
     * Get if the customer is a subscriber of the newsletter
     *
     * @return bool|null
     */
    public function getSubscribeNewsletter()
    {
        return $this->getData('subscribe_newsletter') !== null ? (bool)$this->getData('subscribe_newsletter') : null;
    }

    /**
     * Set DOB
     *
     * @param string $dob
     *
     * @return Customer
     *
     * @throws Exception
     */
    public function setDob($dob)
    {
        return $this->setData('dob', new DateTime($dob));
    }

    /**
     * @inheritDoc
     */
    public function export()
    {
        $result = [
            'magento_id'       => $this->getMagentoId(),
            'fastmag_id'       => $this->getFastmagId(),
            'store_id'         => $this->getStoreId(),
            'gender'           => $this->getGender(),
            'prefix'           => $this->getPrefix(),
            'firstname'        => $this->getFirstname(),
            'lastname'         => $this->getLastname(),
            'email_address'    => $this->getEmailAddress(),
            'dob'              => $this->getDob(),
            'subscribe_nl'     => $this->getSubscribeNewsletter(),
            'billing_address'  => ($this->getBillingAddress() ? $this->getBillingAddress()->export() : null),
            'shipping_address' => ($this->getShippingAddress() ? $this->getShippingAddress()->export() : null)
        ];

        return array_filter(
            $result,
            static function ($value) {
                return $value !== null && $value !== [];
            }
        );
    }
}
