<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-27
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker;

use Exception;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Jobqueue as AbstractJob;
use Fastmag\Sync\Model\Jobqueue\StandardRepository as JobRepository;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\Collection;
use Magento\Framework\Exception\CouldNotSaveException;
use Monolog\Logger as Monolog;

/**
 * Trait Standard
 *
 * Trait for all "standard" workers
 */
trait Standard
{
    /** @var Logger $logger */
    protected $logger;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /** @var Collection $jobs */
    protected $jobs;

    /**
     * Get worker code
     *
     * @return string
     */
    abstract public function getCode();

    /**
     * Main method
     *
     * @return void
     *
     * @throws ProcessException
     */
    abstract public function run();

    /**
     * Tells if the current worker has some subordinate workers to be run before it
     *
     * @return bool
     */
    public function hasSubordinateWorkersBefore()
    {
        return property_exists($this, 'subordinateWorkersBefore') && count($this->subordinateWorkersBefore) > 0;
    }

    /**
     * Returns the list of workers codes to be run before the current worker
     *
     * @return string[]
     */
    public function getSubordinateWorkersBefore()
    {
        return (property_exists($this, 'subordinateWorkersBefore') ? $this->subordinateWorkersBefore : []);
    }

    /**
     * Tells if the current worker has some subordinate workers to be run after it
     *
     * @return bool
     */
    public function hasSubordinateWorkersAfter()
    {
        return property_exists($this, 'subordinateWorkersAfter') && count($this->subordinateWorkersAfter) > 0;
    }

    /**
     * Returns the list of workers codes to be run after the current worker
     *
     * @return string[]
     */
    public function getSubordinateWorkersAfter()
    {
        return (property_exists($this, 'subordinateWorkersAfter') ? $this->subordinateWorkersAfter : []);
    }

    /**
     * Get jobs list
     *
     * @return Collection $jobs
     */
    public function getJobs()
    {
        return $this->jobs;
    }

    /**
     * Set jobs list
     *
     * @param Collection $jobs
     *
     * @return self
     */
    public function setJobs($jobs)
    {
        $this->jobs = $jobs;

        return $this;
    }

    /**
     * Invalidate job
     *
     * @param AbstractJob $job
     * @param Exception   $exception
     *
     * @return Standard
     */
    public function invalidateJob($job, $exception)
    {
        $this->jobRepository->invalidate($job, $exception);

        $this->log($job, '[ERROR] ' . $exception->getMessage(), Monolog::ERROR);

        return $this;
    }

    /**
     * Skip job
     *
     * @param AbstractJob $job
     * @param Exception   $exception
     *
     * @return Standard
     */
    public function skipJob($job, $exception)
    {
        $this->jobRepository->skip($job, $exception);

        $this->log($job, '[ERROR] ' . $exception->getMessage(), Monolog::ERROR);

        return $this;
    }

    /**
     * Save job, or throws an exception
     *
     * @param AbstractJob $job
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function saveJob($job)
    {
        try {
            $this->jobRepository->save($job);
        } catch (CouldNotSaveException $exception) {
            $message = __('[ERROR] Unable to save job.');

            $this->log($job, $message->render(), Monolog::CRITICAL);

            throw new ProcessException($message);
        }
    }

    /**
     * Debug message
     *
     * @param AbstractJob $job
     * @param string      $message
     * @param int         $level
     *
     * @return void
     */
    protected function log($job, $message, $level = Monolog::DEBUG)
    {
        $this->logger->addRecord(
            $level,
            '[' . $this->getCode() . '][Job #' . $job->getId() . '][Entity ' . $job->getContentId() . '] ' . $message
        );
    }
}
