<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-10-27
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration;

use Exception;
use Fastmag\Sync\Api\Data\OrderInterfaceFactory as SyncOrderFactory;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\OrderRepositoryInterface as SyncOrderRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration as IntegrationTrait;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Order
 *
 * Abstract class for Order related Integration workers
 */
abstract class Order extends Worker
{
    use IntegrationTrait;

    /** @var Config $config */
    protected $config;

    /** @var SyncOrderFactory $syncOrderFactory */
    protected $syncOrderFactory;

    /** @var SyncOrderRepository $syncOrderRepository */
    protected $syncOrderRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /** @var OrderEntity $currentEntity */
    protected $currentEntity;

    /** @var array $currentJobApiCalls */
    protected $currentJobApiCalls = [];

    /**
     * Integration constructor.
     *
     * @param Logger                $logger
     * @param JobRepository         $jobRepository
     * @param Api                   $api
     * @param Config                $config
     * @param SyncOrderFactory      $syncedOrderFactory
     * @param SyncOrderRepository   $syncedOrderRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param Json                  $jsonSerializer
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Api $api,
        Config $config,
        SyncOrderFactory $syncedOrderFactory,
        SyncOrderRepository $syncedOrderRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Json $jsonSerializer
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->api = $api;
        $this->config = $config;
        $this->syncOrderFactory = $syncedOrderFactory;
        $this->syncOrderRepository = $syncedOrderRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->jsonSerializer = $jsonSerializer;
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_ORDER_EXPORT_ENABLE);
    }

    /**
     * Send cancellation to Fastmag
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendCancellation()
    {
        $objectName = 'Ventes';
        $endpoint = '/boa/sale/cancel/index.ips';

        $type = $this->currentEntity->getTransactionType();
        if ($type === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION_CANCELLATION) {
            $objectName = 'Reservations';
            $endpoint = '/boa/Reservation/cancel/index.ips';
        }

        if ($this->currentEntity->getLastTransaction() === null) {
            throw new JobException(__(
                'Unable to cancel order #%1 to Fastmag through EDI: no previous transaction found.',
                $this->currentEntity->getMagentoId()
            ));
        }

        $lastTransactionId = $this->currentEntity->getLastTransaction()->getTransactionId();

        $request = (object)[$objectName => [(object)['Numero' => $lastTransactionId]]];

        try {
            $this->sendRequest($type, $endpoint, $request, $lastTransactionId);
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to cancel order #%1 on Fastmag through API. Code: %2. Message: %3. Data sent: %4',
                $this->currentEntity->getMagentoId(),
                $exception->getCode(),
                $exception->getMessage(),
                $exception->getRequest()
            ));
        }
    }

    /**
     * Send request to Fastmag through API
     *
     * @param string   $transactionType
     * @param string   $endpoint
     * @param mixed    $request
     * @param int|null $transactionId
     *
     * @return void
     *
     * @throws ApiException
     */
    protected function sendRequest($transactionType, $endpoint, $request, $transactionId = null)
    {
        $callData = [
            'transaction_id'   => $transactionId,
            'transaction_type' => $transactionType,
            'endpoint'         => $endpoint,
            'request'          => $this->jsonSerializer->serialize($request),
            'status'           => null,
            'message'          => null,
            'request_at'       => date('Y-m-d H:i:s'),
            'response_at'      => null,
            'origin_shop'      => null
        ];

        if ($transactionType === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE) {
            $callData['transaction_id'] = $request->Reservations[0]->NumResa;
        }

        $response = $this->api->post($endpoint, $request);

        if ($transactionType !== OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE
            && array_key_exists('data', $response)
        ) {
            $response = $response['data'][0];

            if ($response['status'] === 'OK') {
                $callData['transaction_id'] = $response['Id'];
            }
        }

        $callData['status'] = $response['status'];
        $callData['response_at'] = date('Y-m-d H:i:s');

        if ($response['status'] === 'KO') {
            $message = (array_key_exists('Message', $response) ? $response['Message'] : '');
            $callData['Message'] = $message;

            throw new ApiException(
                __($message),
                $this->api->getLastRequest(),
                null,
                (array_key_exists('errorCode', $response) ? $response['errorCode'] : 0)
            );
        }

        $this->currentJobApiCalls[] = $callData;
    }

    /**
     * Save order synchronizations
     *
     * @param string $requestAt
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveSyncOrder($requestAt)
    {
        foreach ($this->currentJobApiCalls as $apiCall) {
            $syncedOrder = $this->syncOrderFactory->create();

            $syncedOrder->setOrderId($this->currentEntity->getMagentoId())
                ->setTransactionId($apiCall['transaction_id'])
                ->setTransactionType($apiCall['transaction_type'])
                ->setRequest($apiCall['request'])
                ->setEndpoint($apiCall['endpoint'])
                ->setStatus($apiCall['status'])
                ->setRequestAt($requestAt)
                ->setResultAt($apiCall['response_at'])
                ->setOriginShop($apiCall['origin_shop'])
                ->setTargetShop($this->currentEntity->getFastmagShop());

            try {
                $this->syncOrderRepository->save($syncedOrder);
            } catch (Exception $exception) {
                throw new JobException(
                    __('Unable to save order #%1 synchronization on Magento DB', $this->currentEntity->getMagentoId())
                );
            }
        }
    }
}
