<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-10-28
 ******************************************************************************/

namespace Fastmag\Sync\Model\Jobqueue;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterfaceFactory as JobFactory;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoSearchResultsInterfaceFactory as SearchResultsFactory;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToMagento as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToMagento\CollectionFactory;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SortOrder;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Reflection\DataObjectProcessor;

/**
 * Class ToMagentoRepository
 *
 * Repository of jobs used for Fastmag to Magento synchronization
 */
class ToMagentoRepository extends StandardRepository implements ToMagentoRepositoryInterface
{
    /** @var ResourceModel $resource */
    protected $resource;

    /** @var ToFastmagFactory $jobFactory */
    protected $jobFactory;

    /** @var CollectionFactory $collectionFactory */
    protected $collectionFactory;

    /** @var SearchResultsFactory $searchResultsFactory */
    protected $searchResultsFactory;

    /** @var DataObjectHelper $dataObjectHelper */
    protected $dataObjectHelper;

    /** @var DataObjectProcessor $dataObjectProcessor */
    protected $dataObjectProcessor;

    /** @var JobFactory $jobInterfaceFactory */
    protected $jobInterfaceFactory;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var FilterBuilder $filterBuilder */
    protected $filterBuilder;

    /** @var CollectionProcessorInterface */
    private $collectionProcessor;

    /**
     * ToMagentoRepository constructor
     *
     * @param ResourceModel                     $resource
     * @param ToMagentoFactory                  $jobFactory
     * @param JobFactory                        $jobInterfaceFactory
     * @param CollectionFactory                 $collectionFactory
     * @param SearchResultsFactory              $searchResultsFactory
     * @param DataObjectHelper                  $dataObjectHelper
     * @param DataObjectProcessor               $dataObjectProcessor
     * @param SearchCriteriaBuilder             $searchCriteriaBuilder
     * @param FilterBuilder                     $filterBuilder
     * @param CollectionProcessorInterface|null $collectionProcessor
     */
    public function __construct(
        ResourceModel $resource,
        ToMagentoFactory $jobFactory,
        JobFactory $jobInterfaceFactory,
        CollectionFactory $collectionFactory,
        SearchResultsFactory $searchResultsFactory,
        DataObjectHelper $dataObjectHelper,
        DataObjectProcessor $dataObjectProcessor,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        FilterBuilder $filterBuilder,
        CollectionProcessorInterface $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->jobFactory = $jobFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->dataObjectHelper = $dataObjectHelper;
        $this->jobInterfaceFactory = $jobInterfaceFactory;
        $this->dataObjectProcessor = $dataObjectProcessor;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->filterBuilder = $filterBuilder;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(Job $job)
    {
        try {
            $this->resource->save($job);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getById($jobId)
    {
        $job = $this->jobFactory->create();
        $this->resource->load($job, $jobId);
        if (!$job->getId()) {
            throw new NoSuchEntityException(__('The job with the "%1" ID doesn\'t exist.', $jobId));
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getByUniqueKey($contentId, $jobCode, $priority)
    {
        $job = $this->jobFactory->create();
        $this->resource->loadByUniqueKey($job, $contentId, $jobCode, $priority);
        if (!$job->getId()) {
            throw new NoSuchEntityException(
                __('The job with the following data doesn\'t exist : "%1", "%2", "%3"', $contentId, $jobCode, $priority)
            );
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $criteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($criteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function getListByPriorityAndCode(SearchCriteriaInterface $criteria, $limit)
    {
        $searchResultsArray = [];

        // Get priorities and job_code available
        $prioritiesCollection = $this->collectionFactory->create();

        $this->collectionProcessor->process($criteria, $prioritiesCollection);

        $prioritiesCollection->getSelect()->reset(Select::COLUMNS)
            ->columns([Job::PRIORITY, Job::JOB_CODE])
            ->distinct()
            ->order(Job::PRIORITY . ' ' . Select::SQL_DESC);

        /** @var Job $row */
        foreach ($prioritiesCollection as $row) {
            $collection = $this->collectionFactory->create();

            $this->searchCriteriaBuilder->setFilterGroups((array)$criteria->getFilterGroups())
                ->addFilter(Job::PRIORITY, $row->getPriority())
                ->addFilter(Job::JOB_CODE, $row->getJobCode());

            $rowCriteria = $this->searchCriteriaBuilder->create();

            $this->collectionProcessor->process($rowCriteria, $collection);

            $collection->setOrder(Job::JOB_ID, SortOrder::SORT_ASC)
                ->setPageSize($limit)
                ->setCurPage(1);

            $searchResultsArray[$row->getPriority()][$row->getJobCode()] = $collection;
        }

        return $searchResultsArray;
    }

    /**
     * @inheritDoc
     */
    public function delete(Job $job)
    {
        try {
            $this->resource->delete($job);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($jobId)
    {
        return $this->delete($this->getById($jobId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessorInterface
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
