<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-22
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Customer;

use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Data\CustomerInterfaceFactory as SyncedCustomerFactory;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Customer;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Model\ScopeInterface;

/**
 * Class Deletion
 *
 * Integration class used for deleting customers from Magento to Fastmag
 */
class Deletion extends Customer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_customer_deletion';

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /**
     * Integration constructor.
     *
     * @param Logger                   $logger
     * @param JobRepository            $jobRepository
     * @param Api                      $api
     * @param FastmagSql               $fastmagSql
     * @param Config                   $config
     * @param SyncedCustomerFactory    $syncedCustomerFactory
     * @param SyncedCustomerRepository $syncedCustomerRepository
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param Json                     $jsonSerializer
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Api $api,
        FastmagSql $fastmagSql,
        Config $config,
        SyncedCustomerFactory $syncedCustomerFactory,
        SyncedCustomerRepository $syncedCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Json $jsonSerializer
    ) {
        parent::__construct(
            $logger,
            $jobRepository,
            $api,
            $fastmagSql,
            $config,
            $syncedCustomerFactory,
            $syncedCustomerRepository,
            $searchCriteriaBuilder
        );

        $this->jsonSerializer = $jsonSerializer;
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_DELETION_ENABLE);
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        $customerId = (int)$job->getContentId();
        $fastmagId = false;

        // TODO: change (un)serializer logic for jobs to always get unserialized data and save serialized data
        $jobHydratedData = $this->jsonSerializer->unserialize($job->getHydratedData());

        if (!array_key_exists('fastmag_id', $jobHydratedData) || $jobHydratedData['fastmag_id'] === null) {
            $fastmagId = $this->getFastmagIdByEmail($jobHydratedData['email_adress']);

            if ($fastmagId) {
                $jobHydratedData['fastmag_id'] = $fastmagId;

                $this->saveSyncCustomer($customerId, $fastmagId);
            }
        }

        if ($fastmagId) {
            $this->sendInstructionToFastmag($customerId, $jobHydratedData);
            $this->saveSyncCustomer($customerId, $jobHydratedData['fastmag_id'], true);
        }
    }

    /**
     * Send CONSIGNE for the customer deletion to Fastmag
     *
     * @param int   $magentoId
     * @param array $hydratedData
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendInstructionToFastmag($magentoId, $hydratedData)
    {
        $fastmagShop = $this->config->getValue(
            Config::XML_PATH_ORDER_WORKFLOW_SHOP,
            ScopeInterface::SCOPE_STORES,
            $hydratedData['store_id']
        );

        $data = (object)['Consignes' => [(object)[
            'Destinataire' => $fastmagShop,
            'Sujet'        => 'SUPPRESSION COMPTE CLIENT',
            'Message'      => 'Bonjour,{CR}{LF}Le client ayant l\'ID Fastmag #' . $hydratedData['fastmag_id']
                . ' a demande a etre supprime dans Fastmag, conformement au RGPD.',
            'Expediteur'   => $this->config->getValue(Config::XML_PATH_CONNECT_API_SHOP),
            'Slide'        => '1'
        ]]];

        try {
            $response = $this->api->post('/boa/consigne/create/index.ips', $data);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('message', $response) ? $response['message'] : '');

                throw new ApiException(__($message), $this->api->getLastRequest());
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send instruction for the deletion of the customer #%1 on Fastmag through API. Code: %2. Message: %3. Data sent: %4',
                $magentoId,
                $exception->getCode(),
                $exception->getMessage(),
                $exception->getRequest()
            ));
        }
    }
}
