<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-01-20
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Order;

use Exception;
use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Data\CustomerInterface as SyncedCustomerInterface;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\CustomerFactory as CustomerEntityFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer as AbstractCustomer;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Newsletter\Model\Subscriber;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\OrderRepositoryInterface as OrderRepository;

class Customer extends AbstractCustomer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_order_customer';

    /** @var OrderRepository $orderRepository */
    protected $orderRepository;

    /** @var Subscriber $subscriber */
    protected $subscriber;

    /** @var array $ordersWithCustomers */
    protected $ordersWithCustomers = [];

    /** @var array $ordersPassedAsGuest */
    protected $ordersPassedAsGuest = [];

    /**
     * Customer constructor.
     *
     * @param Logger                   $logger
     * @param JobRepository            $jobRepository
     * @param Config                   $config
     * @param Json                     $jsonSerializer
     * @param SyncedCustomerRepository $syncedCustomerRepository
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param CustomerRepository       $customerRepository
     * @param CustomerEntityFactory    $customerEntityFactory
     * @param OrderRepository          $orderRepository
     * @param Subscriber               $subscriber
     *
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        SyncedCustomerRepository $syncedCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CustomerRepository $customerRepository,
        CustomerEntityFactory $customerEntityFactory,
        Subscriber $subscriber,
        OrderRepository $orderRepository
    ) {
        parent::__construct(
            $logger,
            $jobRepository,
            $config,
            $jsonSerializer,
            $syncedCustomerRepository,
            $searchCriteriaBuilder,
            $customerRepository,
            $customerEntityFactory,
            $subscriber
        );

        $this->orderRepository = $orderRepository;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->getDataFromMagento();
        } catch (Exception $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }

        if (is_array($this->entities)) {
            foreach ($this->getJobs()->getItems() as $job) {
                /** @var OrderEntity $orderEntity */
                $orderEntity = $job->getEntity();

                if (isset($this->entities[$orderEntity->getMagentoId()])) {
                    $orderEntity->setCustomer($this->entities[$orderEntity->getMagentoId()]);

                    $this->hydrateJob($job, $orderEntity);
                    $this->log($job, 'Order\'s customer hydrated');
                }
            }
        }

        foreach ($this->getJobs()->getItems() as $job) {
            $this->saveJob($job);
        }
    }

    /**
     * @inheritDoc
     *
     * @return CustomerEntity[]
     *
     * @throws JobException
     * @throws ProcessException
     */
    protected function getDataFromMagento()
    {
        $this->getCustomerData();

        if (count($this->ordersWithCustomers) > 0) {
            foreach ($this->ordersWithCustomers as $orderId => $customerId) {
                $customerEntity = $this->customerEntityFactory->create();
                $customerEntity->setMagentoId($customerId);
                $this->entities[$orderId] = $customerEntity;
            }

            $customerIds = array_values($this->ordersWithCustomers);

            $this->getSyncedCustomers($customerIds);
            $this->getMagentoCustomersData($customerIds);
        }

        if (count($this->ordersPassedAsGuest) > 0) {
            foreach (array_keys($this->ordersPassedAsGuest) as $contentId) {
                $customerEntity = $this->customerEntityFactory->create();
                $this->entities[$contentId] = $customerEntity;
            }

            $this->generateFakeCustomerEntities(array_values($this->ordersPassedAsGuest));
        }

        return $this->entities;
    }

    /**
     * Retrieve the customer IDs for the orders' jobs
     *
     * @return void
     */
    protected function getCustomerData()
    {
        foreach ($this->getJobs()->getItems() as $job) {
            /** @var OrderEntity $orderEntity */
            $orderEntity = $job->getEntity();

            $transactionType = $orderEntity->getTransactionType();
            $orderId = $orderEntity->getMagentoId();

            if ($orderId !== null && $this->useCustomerData($transactionType)) {
                if ($orderEntity->getIsPassedAsGuest()) {
                    $this->ordersPassedAsGuest[$orderId] = $orderId;
                } else {
                    $customerId = $orderEntity->getCustomerId();
                    if ($customerId === null) {
                        continue;
                    }

                    $this->ordersWithCustomers[$orderId] = $customerId;
                }
            }
        }
    }

    /**
     * Override to get the job with the first associated order of the customer
     *
     * @inheritDoc
     */
    protected function getSyncedCustomers($customerIds)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(SyncedCustomerInterface::MAGENTO_CUSTOMER_ID, $customerIds, 'in')
            ->create();

        $syncedCustomerList = $this->syncedCustomerRepository->getList($searchCriteria);

        foreach ($syncedCustomerList->getItems() as $syncedCustomer) {
            $job = $this->getJob(array_search(
                $syncedCustomer->getMagentoCustomerId(),
                $this->ordersWithCustomers,
                true
            ));

            if ($job === null) {
                throw new JobException(
                    __('There\'s no order job for customer ID #%1', $syncedCustomer->getMagentoCustomerId())
                );
            }

            $this->log($job, 'Customer already synced, Fastmag ID: ' . $syncedCustomer->getFastmagCustomerId());

            $currentEntity = $this->getCustomerEntity($syncedCustomer->getMagentoCustomerId());
            $currentEntity->setFastmagId($syncedCustomer->getFastmagCustomerId());
        }
    }

    /**
     * @inheritDoc
     */
    protected function getCustomerEntity($customerId)
    {
        return $this->entities[array_search($customerId, $this->ordersWithCustomers, true)];
    }

    /**
     * Returns true if the transaction use customer data
     *
     * @param string|null $transactionType
     *
     * @return bool
     */
    protected function useCustomerData($transactionType)
    {
        return $transactionType === null || in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_ORDER,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE
            ],
            true
        );
    }

    /**
     * Generate customer entities with the data of the orders
     *
     * @param int[] $orderIds
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function generateFakeCustomerEntities($orderIds)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(OrderInterface::ENTITY_ID, $orderIds, 'in')
            ->create();

        $ordersList = $this->orderRepository->getList($searchCriteria);

        try {
            foreach ($ordersList->getItems() as $order) {
                $currentEntity = $this->entities[$order->getEntityId()];

                $billingAddress = $order->getBillingAddress();
                if ($billingAddress !== null) {
                    $currentEntity->setPrefix($billingAddress->getPrefix())
                        ->setLastname($billingAddress->getLastname())
                        ->setFirstname($billingAddress->getFirstname())
                        ->setEmailAddress($order->getCustomerEmail())
                        ->setStoreId($order->getStoreId())
                        ->setSubscribeNewsletter(false);
                }
            }
        } catch (Exception $exception) {
            throw new ProcessException(__(
                'Error when hydrating orders passed as guest. Message: %1. Orders IDs: %2',
                $exception->getMessage(),
                implode(', ', $orderIds)
            ));
        }
    }
}
