<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-01-20
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Exception\ProductNotSyncableException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Product\VariationFactory as VariationEntityFactory;
use Fastmag\Sync\Process\Entity\ToMagento\ProductFactory as ProductEntityFactory;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use InvalidArgumentException;
use Magento\Catalog\Helper\Data as CatalogHelper;
use Magento\Catalog\Model\ProductFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\InventoryApi\Api\StockRepositoryInterface as StockRepository;
use Magento\Store\Api\StoreRepositoryInterface as StoreRepository;
use Magento\Store\Model\ScopeInterface;
use Monolog\Logger as Monolog;

/**
 * Class Product
 *
 * Hydration class used for inserting or updating products from Fastmag to Magento
 */
abstract class Product extends Hydration
{
    /** @var ProductFactory $productFactory */
    protected $productFactory;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var StoreRepository $storeRepository */
    protected $storeRepository;

    /** @var ProductEntityFactory $productEntityFactory */
    protected $productEntityFactory;

    /** @var VariationEntityFactory $variationEntityFactory */
    protected $variationEntityFactory;

    /** @var StockRepository $stockRepository */
    protected $stockRepository;

    /** @var Job $currentJob */
    protected $currentJob;

    /** @var ProductEntity $currentEntity */
    protected $currentEntity;

    /** @var array $syncConfig */
    protected $syncConfig;

    /** @var string[] $syncConfigFields */
    protected $syncConfigFields = [
        Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS,
        Config::XML_PATH_PRICE_IMPORT_FASTMAG_RATE,
        Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_LANGUAGE,
        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_REFERENCE_STOCK,
        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS,
        Config::XML_PATH_ORDER_WORKFLOW_SHOP
    ];

    /**
     * Product constructor
     *
     * @param Logger                    $logger
     * @param JobRepository             $jobRepository
     * @param Config                    $config
     * @param Json                      $jsonSerializer
     * @param FastmagSql                $fastmagSql
     * @param ProductFactory            $productFactory
     * @param SearchCriteriaBuilder     $searchCriteriaBuilder
     * @param StoreRepository           $storeRepository
     * @param ProductEntityFactory      $productEntityFactory
     * @param VariationEntityFactory    $variationEntityFactory
     * @param StockRepository           $stockRepository
     *
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        FastmagSql $fastmagSql,
        ProductFactory $productFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        StoreRepository $storeRepository,
        ProductEntityFactory $productEntityFactory,
        VariationEntityFactory $variationEntityFactory,
        StockRepository $stockRepository
    ) {
        parent::__construct($logger, $jobRepository, $config, $jsonSerializer, $fastmagSql);

        $this->productFactory = $productFactory;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->storeRepository = $storeRepository;
        $this->productEntityFactory = $productEntityFactory;
        $this->variationEntityFactory = $variationEntityFactory;
        $this->stockRepository = $stockRepository;
    }

    /**
     * Hydrate the product of the current job
     *
     * @return ProductEntity
     *
     * @throws JobException
     * @throws ProductNotSyncableException
     */
    protected function getDataFromFastmag()
    {
        $fastmagRef = $this->currentJob->getContentId();

        if ($this->isPack($fastmagRef)) {
            throw new ProductNotSyncableException(__(
                'Product "%1" is a pack, which can not be imported with Fastmag_Sync module',
                $fastmagRef
            ));
        }

        if ($this->isUtilitarianProduct($fastmagRef)) {
            throw new ProductNotSyncableException(__(
                'Product "%1" is an utilitarian product, it must not be synced with Magento',
                $fastmagRef
            ));
        }

        $this->initiateCurrentEntity($fastmagRef);

        $this->getParentData();

        if ($this->productMustBeSynced()) {
            $this->getChildrenData();

            if (array_key_exists('stores', $this->syncConfig)) {
                $this->getDataByStore();
            }

            if ($this->currentEntity->getVariations() === null) {
                throw new ProductNotSyncableException(
                    __('No existing or salable variation on Fastmag for the product "%1"', $fastmagRef)
                );
            }
        } else {
            throw new ProductNotSyncableException(__(
                'Product "%1" can\'t be synced, it must be inactive or not in stock: %2',
                $fastmagRef,
                $this->jsonSerializer->serialize($this->currentEntity->export())
            ));
        }

        return $this->currentEntity;
    }

    /**
     * Initiate current product entity
     *
     * @param string $fastmagRef
     *
     * @return void
     */
    protected function initiateCurrentEntity($fastmagRef)
    {
        $this->currentEntity = $this->productEntityFactory->create();
        $this->currentEntity->setRef($fastmagRef)
            ->setMagentoId($this->getMagentoProductId($fastmagRef));
    }

    /**
     * Get product data for each store
     *
     * @return void
     *
     * @throws JobException
     */
    protected function getDataByStore()
    {
        foreach (array_keys($this->syncConfig['stores']) as $storeId) {
            $this->getChildrenData($storeId);
        }
    }

    /**
     * Check if product is a pack
     *
     * @param string $fastmagRef
     *
     * @return bool
     */
    protected function isPack($fastmagRef)
    {
        return (strpos($fastmagRef, 'PACK_') !== false);
    }

    /**
     * Check if product is an utilitarian product (as FP or REGVAT)
     *
     * @param string $fastmagRef
     *
     * @return bool
     */
    protected function isUtilitarianProduct($fastmagRef)
    {
        return in_array($fastmagRef, [Constants::PRODUCT_FP_SKU, Constants::PRODUCT_REGVAT_SKU], true);
    }

    /**
     * Get Magento product ID for Fastmag ref given
     *
     * @param string $fastmagRef
     *
     * @return int|false
     */
    protected function getMagentoProductId($fastmagRef)
    {
        $product = $this->productFactory->create();

        return $product->getIdBySku($fastmagRef);
    }

    /**
     * Get product basic data on Fastmag (VisibleWeb, Actif, creation date, inventory data, family)
     *
     * @return void
     *
     * @throws JobException
     */
    abstract protected function getParentData();

    /**
     * Check if products fill all the condition to be synced
     *
     * @return bool
     */
    protected function productMustBeSynced()
    {
        $result = ($this->currentEntity->getActive() && $this->currentEntity->getVisibleWeb())
            || !$this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ACTIVE_VISIBLEWEB);

        $onlyDefinedStocks = $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS);
        if ($onlyDefinedStocks && $this->currentEntity->getStockLevel() === 0) {
            $result = false;
        }

        if ($this->currentEntity->getMagentoId()) {
            $result = true;
        }

        return $result;
    }

    /**
     * Get config on scope global and store for product sync
     *
     * @return void
     */
    protected function getSyncConfig()
    {
        $defaultValues = [];
        foreach ($this->syncConfigFields as $field) {
            $defaultValues[$field] = $this->getSyncConfigValue($field);
        }

        $result['default'] = $defaultValues;

        $storesList = $this->storeRepository->getList();

        $pricePerWebsite = $this->config->getValue(CatalogHelper::XML_PATH_PRICE_SCOPE);
        $this->logger->debug(
            __('[%1] Price scope: %2', $this->getCode(), ($pricePerWebsite === 0 ? 'global' : 'website'))->render()
        );

        $useMsi = $this->useMSI();
        $this->logger->debug(__('[%1] MSI: %2', $this->getCode(), ($useMsi ? 'enabled' : 'disabled'))->render());

        foreach ($storesList as $store) {
            $storeValues = [];

            foreach ($this->syncConfigFields as $field) {
                $storeValues[$field] = $this->getSyncConfigValue($field, $store->getId());

                if ($field === Config::XML_PATH_PRICE_IMPORT_FASTMAG_RATE && $pricePerWebsite === false) {
                    $storeValues[$field] = $defaultValues[$field];
                }

                if ($useMsi === false
                    && in_array($field, [
                        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_REFERENCE_STOCK,
                        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS
                    ], true)
                ) {
                    $storeValues[$field] = $defaultValues[$field];
                }
            }

            if ($defaultValues !== $storeValues) {
                $result['stores'][$store->getId()] = $storeValues;
            }
        }

        $this->logger->debug(
            __('[%1] Synchronization config: %2', $this->getCode(), $this->jsonSerializer->serialize($result))->render()
        );

        $this->syncConfig = $result;
    }

    /**
     * Get usefull config field value, for scope global, or for a specific store
     *
     * @param string   $field
     * @param int|null $storeId
     *
     * @return mixed
     */
    protected function getSyncConfigValue($field, $storeId = null)
    {
        $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT;

        if ($storeId !== null) {
            $scope = ScopeInterface::SCOPE_STORES;
        }

        $result = $this->config->getValue($field, $scope, $storeId);

        if ($field === Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS) {
            if ($result === null) {
                $result = [];
            } elseif (strpos($result, '|') === false) {
                $result = [$result];
            } else {
                $result = explode('|', $result);
                sort($result);
            }
        }

        return $result;
    }

    /**
     * Check if website use MSI
     *
     * @return bool
     */
    protected function useMsi()
    {
        $searchCriteria = $this->searchCriteriaBuilder->create();

        $stocksList = $this->stockRepository->getList($searchCriteria);

        return $stocksList->getTotalCount() > 1;
    }

    /**
     * Get sync config from syncConfig array
     *
     * @param string   $field
     * @param int|null $storeId
     * @param bool     $default
     */
    protected function getSyncConfigValueFromArray($field, $storeId = null, $default = true)
    {
        $result = '';

        if ($storeId === null && $default === true) {
            $result = $this->syncConfig['default'][$field];
        }

        if ($storeId !== null && array_key_exists($storeId, $this->syncConfig['stores'])) {
            $result = $this->syncConfig['stores'][$storeId][$field];
        }

        return $result;
    }

    /**
     * Get static data for the product from Fastmag
     *
     * @param int|null $storeId
     *
     * @return void
     *
     * @throws JobException
     */
    protected function getChildrenData($storeId = null)
    {
        try {
            $sql = $this->getChildrenDataSql($storeId);
            $this->log($this->currentJob, 'SQL children query: ' . $sql);

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if (count($rows) === 0) {
                $rows = $this->getFastmagSqlConnection()->get(str_replace('stock.AR = 1', 'stock.AR = 0', $sql));
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        if (count($rows) > 0) {
            foreach ($rows as $row) {
                $this->saveChildrenData($row, $storeId);
            }
        }
    }

    /**
     * Get SQL query getting the children data
     *
     * @param int|null $storeId
     *
     * @return string
     *
     * @throws JobException
     * @throws NoConnectionException
     */
    abstract protected function getChildrenDataSql($storeId = null);

    /**
     * Get the condition to use only defined stocks levels to check if a product is in stock or not
     *
     * @param int|null $storeId
     *
     * @return string
     */
    protected function getOnlyDefinedStocksCondition($storeId = null)
    {
        $result = '';

        if ($this->getSyncConfigValueFromArray(Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS, $storeId)) {
            $result = 'HAVING active_stock = 1 ';
        }

        return $result;
    }

    /**
     * Get the condition to use stocks codes defined in config
     *
     * @param int|null $storeId
     *
     * @return string
     *
     * @throws NoConnectionException
     */
    protected function getStocksCodesCondition($storeId = null)
    {
        $result = '';

        $referenceStocks = $this->getReferenceStocks($storeId);

        if ($referenceStocks !== null && count($referenceStocks) !== 0) {
            $result = ' AND CodeMag IN (' . $this->getFastmagSqlConnection()->escape($referenceStocks) . ')';
        }

        return $result;
    }

    /**
     * Set all stock references used for synchronization
     *
     * @param int|null $storeId
     *
     * @return array
     */
    protected function getReferenceStocks($storeId = null)
    {
        $result = [];

        $result[] = $this->getSyncConfigValueFromArray(
            Config::XML_PATH_INVENTORY_FASTMAG_STOCK_REFERENCE_STOCK,
            $storeId
        );
        $alternativesStocks = $this->getSyncConfigValueFromArray(
            Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS,
            $storeId
        );

        return array_unique(array_merge($result, $alternativesStocks));
    }

    /**
     * Save static data into specific entity
     *
     * @param array    $row
     * @param int|null $storeId
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveChildrenData($row, $storeId = null)
    {
        $row = $this->extractJsonData($row);

        $variation = $this->variationEntityFactory->create();

        if ($storeId === null) {
            $variation->addData($row);

            $this->currentEntity->addVariation($variation, $row['color'], $row['size']);
        } else {
            $currentVariation = $this->currentEntity->getVariation($row['color'], $row['size']);

            if ($currentVariation === null) {
                throw new JobException(__(
                    'No variation already existing for ref "%1", color "%2", size "%3"',
                    $this->currentEntity->getRef(),
                    $row['color'],
                    $row['size']
                ));
            }

            foreach ($row as $code => $value) {
                // Set i18n values only if it's different of the default one, except for price,
                // which has to be set for each rule to distinguish default value from no value at all
                if ($value !== ''
                    && (stripos('price', $code) !== false || $value !== $currentVariation->getData($code))
                ) {
                    $variation->setData($code, $value);
                }
            }

            if ($variation->hasData()) {
                $currentVariation->addI18n($variation, $storeId);
            }
        }
    }

    /**
     * Returns subquery for stock price (standard price)
     *
     * @param string $fastmagShop
     * @param string $priceRateCode
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getStandardPriceSubquery($fastmagShop, $priceRateCode)
    {
        try {
            if ($priceRateCode === 'prix_indicatif') {
                $result = 'pf.PrixVente';
            } else {
                $result = 'SELECT stpr.PrixVente
                    FROM stock AS stpr
                    WHERE stpr.CodeMag = ' . $this->getFastmagSqlConnection()->escape($fastmagShop) . '
                        AND stpr.AR = 1
                        AND stpr.BarCode = stock.BarCode
                        AND stpr.Couleur = stock.Couleur
                        AND stpr.Taille = stock.Taille
                        AND stpr.PrixVente > 0
                    ORDER BY stpr.Date, stpr.ID
                    LIMIT 1';
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * Returns subquery for combination price
     *
     * @param string $priceRateCode
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getCombinationPriceSubquery($priceRateCode)
    {
        try {
            if ($priceRateCode === 'prix_indicatif') {
                $result = 'pf.PrixVente';
            } else {
                $result = 'SELECT p.PrixVente
                    FROM produits AS p
                    WHERE p.BarCode = stock.BarCode
                        AND p.Couleur = stock.Couleur
                        AND p.Taille = stock.Taille
                        AND p.PrixVente > 0';
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * Returns subquery for rate price
     *
     * @param string $rateCode
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getRatePriceSubquery($rateCode)
    {
        try {
            if ($rateCode !== '') {
                if ($rateCode === 'prix_indicatif') {
                    $result = 'pf.PrixVente';
                } else {
                    $result = 'SELECT pt.Prix
                        FROM produitstarifs AS pt
                        WHERE pt.Tarif = ' . $this->getFastmagSqlConnection()->escape($rateCode) . '
                            AND pt.BarCode = stock.BarCode
                            AND pt.Couleur = stock.Couleur
                            AND pt.Taille = stock.Taille
                            AND pt.Prix > 0
                        ORDER BY pt.Prix DESC
                        LIMIT 1';
                }
            } else {
                $result = '0';
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * Returns subquery for sales price
     *
     * @param string $fastmagShop
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getSalePriceSubquery($fastmagShop)
    {
        try {
            $result = 'SELECT CONCAT(
                        \'{"\', \'sale_price\', \'": \', Prix,
                        \', "\', \'sale_discount\', \'": \', Remise,
                        \', "\', \'sale_begin_at\', \'": "\', DateDebut,
                        \'", "\', \'sale_end_at\', \'": "\', DateFin,
                        \'", "\', \'sale_reason\', \'": "\', Motif, \'"}\'
                    )
                FROM prixremise
                WHERE CodeMag = ' . $this->getFastmagSqlConnection()->escape($fastmagShop) . '
                    AND BarCode = stock.BarCode
                    AND DateDebut <= date(NOW())
                    AND DateFin >= date(NOW())
                ORDER BY DateDebut DESC, DateFin DESC, Remise, Prix
                LIMIT 1';
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * Returns subquery for size and color sales price
     *
     * @param string $fastmagShop
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getVariationSalePriceSubquery($fastmagShop)
    {
        try {
            $result = 'SELECT CONCAT(
                        \'{"\', \'variation_sale_price\', \'": \', Prix,
                        \', "\', \'variation_sale_discount\', \'": \', Remise,
                        \', "\', \'variation_sale_begin_at\', \'": "\', DateDebut,
                        \'", "\', \'variation_sale_end_at\', \'": "\', DateFin,
                        \'", "\', \'variation_sale_reason\', \'": "\', Motif, \'"}\'
                    )
                FROM prixremisetc AS ptc
                WHERE ptc.CodeMag = ' . $this->getFastmagSqlConnection()->escape($fastmagShop) . '
                    AND ptc.BarCode = stock.BarCode
                    AND ptc.Couleur = stock.Couleur
                    AND ptc.Taille = stock.Taille
                    AND (ptc.Remise > 0 OR ptc.Prix > 0)
                    AND DateDebut <= date(NOW())
                    AND ptc.DateFin >= date(NOW())
                ORDER BY ptc.DateDebut DESC, ptc.DateFin DESC, ptc.Remise, ptc.Prix
                LIMIT 1';
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * Returns subquery for buying price
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getBuyingPriceSubquery()
    {
        try {
            $result = 'SELECT prod.PrixAchat
                FROM produits AS prod
                WHERE prod.BarCode = stock.BarCode
                    AND prod.Couleur = stock.Couleur
                    AND prod.Taille = stock.Taille
                LIMIT 1';
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * Extract JSON data and merge them to the row given
     *
     * @param array $row
     *
     * @return array
     */
    protected function extractJsonData($row)
    {
        $fieldsToExtract = ['combination_data', 'sale_price', 'variation_sale_price'];
        $fieldsExtracted = [];

        foreach ($fieldsToExtract as $field) {
            if (array_key_exists($field, $row) && $row[$field] !== null) {
                try {
                    $fieldsExtracted[] = $this->jsonSerializer->unserialize($row[$field]);
                    unset($row[$field]);
                } catch (InvalidArgumentException $exception) {
                    $this->log(
                        $this->currentJob,
                        'Error when unserializing ' . $field . ' from Fastmag: ' . $row[$field],
                        Monolog::WARNING
                    );
                }
            }
        }

        return array_merge($row, ...$fieldsExtracted);
    }
}
