<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-01-25
 ******************************************************************************/

namespace Fastmag\Sync\Model\Jobqueue;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoScheduledInterface;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoScheduledInterfaceFactory;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoScheduledSearchResultsInterfaceFactory;
use Fastmag\Sync\Api\Jobqueue\ToMagentoScheduledRepositoryInterface;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToMagentoScheduled as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToMagentoScheduled\CollectionFactory;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Reflection\DataObjectProcessor;

/**
 * Class ToMagentoScheduledRepository
 *
 * Repository of jobs used for scheduled Fastmag to Magento synchronization
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class ToMagentoScheduledRepository implements ToMagentoScheduledRepositoryInterface
{
    /** @var ResourceModel $resource */
    protected $resource;

    /** @var ToFastmagFactory $jobFactory */
    protected $jobFactory;

    /** @var CollectionFactory $collectionFactory */
    protected $collectionFactory;

    /** @var ToMagentoScheduledSearchResultsInterfaceFactory $searchResultsFactory */
    protected $searchResultsFactory;

    /** @var DataObjectHelper $dataObjectHelper */
    protected $dataObjectHelper;

    /** @var DataObjectProcessor $dataObjectProcessor */
    protected $dataObjectProcessor;

    /** @var ToMagentoScheduledInterfaceFactory $jobInterfaceFactory */
    protected $jobInterfaceFactory;

    /** @var CollectionProcessorInterface $collectionProcessor */
    private $collectionProcessor;

    /**
     * ToMagentoScheduledRepository constructor
     *
     * @param ResourceModel                                   $resource
     * @param ToMagentoScheduledFactory                       $jobFactory
     * @param ToMagentoScheduledInterfaceFactory              $jobInterfaceFactory
     * @param CollectionFactory                               $collectionFactory
     * @param ToMagentoScheduledSearchResultsInterfaceFactory $searchResultsFactory
     * @param DataObjectHelper                                $dataObjectHelper
     * @param DataObjectProcessor                             $dataObjectProcessor
     * @param CollectionProcessorInterface|null               $collectionProcessor
     */
    public function __construct(
        ResourceModel $resource,
        ToMagentoScheduledFactory $jobFactory,
        ToMagentoScheduledInterfaceFactory $jobInterfaceFactory,
        CollectionFactory $collectionFactory,
        ToMagentoScheduledSearchResultsInterfaceFactory $searchResultsFactory,
        DataObjectHelper $dataObjectHelper,
        DataObjectProcessor $dataObjectProcessor,
        CollectionProcessorInterface $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->jobFactory = $jobFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->dataObjectHelper = $dataObjectHelper;
        $this->jobInterfaceFactory = $jobInterfaceFactory;
        $this->dataObjectProcessor = $dataObjectProcessor;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(ToMagentoScheduledInterface $job)
    {
        try {
            $this->resource->save($job);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getById($jobId)
    {
        $job = $this->jobFactory->create();
        $this->resource->load($job, $jobId);
        if (!$job->getId()) {
            throw new NoSuchEntityException(__('The job with the "%1" ID doesn\'t exist.', $jobId));
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $criteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($criteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(ToMagentoScheduledInterface $job)
    {
        try {
            $this->resource->delete($job);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($jobId)
    {
        return $this->delete($this->getById($jobId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessorInterface
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
