<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-01-25
 ******************************************************************************/

namespace Fastmag\Sync\Plugin\Jobqueue\ToFastmag;

use Exception;
use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Jobqueue\ToFastmagFactory as JobFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\CustomerFactory as CustomerEntityFactory;
use Fastmag\Sync\Plugin\Jobqueue\ToFastmag;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Customer\Api\Data\CustomerInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * Class CustomerRepositoryPlugin
 *
 * Plugin on CustomerRepositoryInterface
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class CustomerRepositoryPlugin extends ToFastmag
{
    /** @var Config $config */
    protected $config;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var CustomerEntityFactory $customerEntityFactory */
    protected $customerEntityFactory;

    /** @var SyncedCustomerRepository $syncedCustomerRepository */
    protected $syncedCustomerRepository;

    /**
     * Save contructor
     *
     * @param JobFactory               $jobFactory
     * @param JobRepository            $jobRepository
     * @param Logger                   $logger
     * @param Config                   $config
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param CustomerEntityFactory    $customerEntityFactory
     * @param SyncedCustomerRepository $syncedCustomerRepository
     */
    public function __construct(
        JobFactory $jobFactory,
        JobRepository $jobRepository,
        Logger $logger,
        Config $config,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CustomerEntityFactory $customerEntityFactory,
        SyncedCustomerRepository $syncedCustomerRepository
    ) {
        parent::__construct($jobFactory, $jobRepository, $logger);

        $this->config = $config;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->customerEntityFactory = $customerEntityFactory;
        $this->syncedCustomerRepository = $syncedCustomerRepository;
    }

    /**
     * @see CustomerRepository::save()
     *
     * @param CustomerRepository $subject
     * @param CustomerInterface  $customer
     *
     * @return CustomerInterface
     *
     * @throws CouldNotDeleteException
     * @throws CouldNotSaveException
     */
    public function afterSave(CustomerRepository $subject, CustomerInterface $customer)
    {
        $customerId = $customer->getId();

        if ($customerId) {
            $searchCriteria = $this->searchCriteriaBuilder->addFilter(Job::CONTENT_ID, $customerId)
                ->addFilter(Job::JOB_CODE, 'tofastmag_integration_customer_deletion')
                ->create();

            $deletionJobsList = $this->jobRepository->getList($searchCriteria);

            if ($deletionJobsList->getTotalCount() > 0) {
                foreach ($deletionJobsList->getItems() as $deletionJob) {
                    $this->jobRepository->delete($deletionJob);
                }
            }

            $job = $this->jobFactory->create();
            $job->setContentId($customerId)
                ->setJobCode('tofastmag_integration_customer');

            $this->jobRepository->save($job);
        }

        return $customer;
    }

    /**
     * @see CustomerRepository::deleteById()
     *
     * @param CustomerRepository $subject
     * @param int                $customerId
     *
     * @return void
     *
     * @throws CouldNotDeleteException
     * @throws ProcessException
     */
    public function beforeDeleteById(CustomerRepository $subject, int $customerId)
    {
        if ($customerId) {
            $searchCriteria = $this->searchCriteriaBuilder->addFilter(Job::CONTENT_ID, $customerId)
                ->addFilter(Job::JOB_CODE, 'tofastmag_integration_customer')
                ->create();

            $savingJobsList = $this->jobRepository->getList($searchCriteria);

            if ($savingJobsList->getTotalCount() > 0) {
                foreach ($savingJobsList->getItems() as $savingJob) {
                    $this->jobRepository->delete($savingJob);
                }
            }

            $job = $this->jobFactory->create();
            $job->setContentId($customerId)
                ->setJobCode('tofastmag_integration_customer_deletion');

            try {
                $customerData = $this->getDeletedCustomerData($customerId, $subject);

                $job->setHydratedData($customerData);

                $this->jobRepository->save($job);
            } catch (NoSuchEntityException $exception) {
                // Do nothing, as the customer does not exist in Fastmag in the first place
            } catch (Exception $exception) {
                $message = __('Unable to get deleted customer #%1 data, deletion job not created', $customerId);
                $this->logger->error($message);

                throw new ProcessException($message);
            }
        }
    }

    /**
     * Acts as hydration worker to get customer data before its deletion
     *
     * @param int                $customerId
     * @param CustomerRepository $customerRepository
     *
     * @return array
     *
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    protected function getDeletedCustomerData($customerId, $customerRepository)
    {
        $customer = $customerRepository->getById($customerId);
        $syncedCustomer = $this->syncedCustomerRepository->getByMagentoId($customerId);

        $customerEntity = $this->customerEntityFactory->create();
        $customerEntity->setMagentoId($customerId)
            ->setEmailAddress($customer->getEmail())
            ->setFastmagId($syncedCustomer->getFastmagCustomerId())
            ->setStoreId($customer->getStoreId());

        return $customerEntity->export();
    }
}
