<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-01-25
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Order;

use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\AddressFactory as AddressEntityFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Address as AbstractAddress;
use Magento\Customer\Api\AddressRepositoryInterface as AddressRepository;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Sales\Api\OrderRepositoryInterface as OrderRepository;

/**
 * Class Address
 *
 * Hydration class used for inserting or updating orders delivery addresses from Magento to Fastmag
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Address extends AbstractAddress
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_order_address';

    /** @var OrderRepository $orderRepository */
    protected $orderRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var SortOrderBuilder $sortOrderBuilder */
    protected $sortOrderBuilder;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /** @var Job $currentJob */
    protected $currentJob;

    /**
     * Address constructor.
     *
     * @param Logger                $logger
     * @param JobRepository         $jobRepository
     * @param Config                $config
     * @param Json                  $jsonSerializer
     * @param CustomerRepository    $customerRepository
     * @param AddressRepository     $addressRepository
     * @param AddressEntityFactory  $addressEntityFactory
     * @param OrderRepository       $orderRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param SortOrderBuilder      $sortOrderBuilder
     *
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        CustomerRepository $customerRepository,
        AddressRepository $addressRepository,
        AddressEntityFactory $addressEntityFactory,
        OrderRepository $orderRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder
    ) {
        parent::__construct(
            $logger,
            $jobRepository,
            $config,
            $jsonSerializer,
            $customerRepository,
            $addressRepository,
            $addressEntityFactory
        );

        $this->orderRepository = $orderRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        foreach ($this->getJobs()->getItems() as $job) {
            $this->currentJob = $job;

            try {
                if (!$job->isInError()) {
                    $addressEntities = $this->getDataFromMagento();

                    if (count($addressEntities) > 0) {
                        /** @var OrderEntity $orderEntity */
                        $orderEntity = $job->getEntity();
                        $orderEntity->setBillingAddress($addressEntities['billing'])
                            ->setShippingAddress($addressEntities['shipping']);

                        $this->hydrateJob($job, $orderEntity);
                        $this->log($job, 'Order\'s addresses hydrated');
                    }
                }
            } catch (JobException $exception) {
                $this->invalidateJob($job, $exception);
            }

            $this->saveJob($job);
        }
    }

    /**
     * @inheritDoc
     *
     * @return AddressEntity[]
     *
     * @throws JobException
     */
    protected function getDataFromMagento()
    {
        $result = [];

        /** @var OrderEntity $orderEntity */
        $orderEntity = $this->currentJob->getEntity();
        $transactionType = $orderEntity->getTransactionType();

        if ($this->useCustomerData($transactionType)) {
            $customerId = $orderEntity->getCustomerId();
            $orderId = $orderEntity->getMagentoId();

            try {
                /** @throws NoSuchEntityException|InputException */
                $order = $this->orderRepository->get($orderId);
            } catch (NoSuchEntityException $exception) {
                throw new JobException(__('There\'s no order with ID #%1', $orderId));
            } catch (InputException $exception) {
                throw new JobException(__('No order ID given'));
            }

            if ($order) {
                $billingAddress = $this->getAddressEntity($customerId, $order);
                $shippingAddress = $this->getAddressEntity($customerId, $order, 'shipping');

                $result = [
                    'billing'  => $billingAddress,
                    'shipping' => $shippingAddress
                ];
            }
        }

        return $result;
    }

    /**
     * Returns true if the transaction use customer data
     *
     * @param string|null $transactionType
     *
     * @return bool
     */
    protected function useCustomerData($transactionType)
    {
        return $transactionType === null || in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_ORDER,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE
            ],
            true
        );
    }
}
