<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-03-16
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\Item as OrderItemEntity;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration as IntegrationTrait;
use Magento\Framework\Serialize\Serializer\Json;
use stdClass;

abstract class Transfer extends Worker
{
    use IntegrationTrait;

    /** @var Config $config */
    protected $config;

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /** @var string $transactionType */
    protected $transactionType = OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_TRANSFER;

    /** @var Job $currentJob */
    protected $currentJob;

    /**
     * Transfer constructor
     *
     * @param Logger        $logger
     * @param JobRepository $jobRepository
     * @param Api           $api
     * @param Config        $config
     * @param Json          $jsonSerializer
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Api $api,
        Config $config,
        Json $jsonSerializer
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->api = $api;
        $this->config = $config;
        $this->jsonSerializer = $jsonSerializer;
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_ORDER_EXPORT_ENABLE);
    }

    /**
     * Send alert to shop
     *
     * @param string            $originShop
     * @param OrderItemEntity[] $items
     * @param array             $qtyTransferred
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendAlert($originShop, $items, $qtyTransferred = [])
    {
        $message = [];

        foreach ($items as $itemId => $itemEntity) {
            $message[] = 'Ref : ' . $itemEntity->getFastmagBarcode()
                . ' - Taille : ' . $itemEntity->getFastmagSize()
                . ' - Couleur : ' . $itemEntity->getFastmagColor()
                . ' - Quantité : ';

            if (array_key_exists($itemId, $qtyTransferred)) {
                $message[] .= $qtyTransferred[$itemId];
            } else {
                $message[] .= $itemEntity->getQtyOrdered();
            }
        }

        $request = (object)[
            'Consignes' => [(object)[
                'Destinataire' => $this->currentJob->getEntity()->getFastmagShop(),
                'Sujet'        => __('Web order #%1', $this->currentJob->getEntity()->getMagentoId()),
                'Message'      => __('To prepare: %1', implode(' *** ', $message)),
                'Expediteur'   => $originShop,
                'Slide'        => '1'
            ]]
        ];

        try {
            $response = $this->api->post('/boa/consigne/create/index.ips', $request);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException(__($message), $this->api->getLastRequest(), null, $response['errorCode']);
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send alert to shop %1. Code: %2. Message: %3. Cause: %4. Alert sent: %5',
                $this->currentJob->getEntity()->getFastmagShop(),
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                implode(' *** ', $message)
            ));
        }
    }

    /**
     * Save order synchronizations
     *
     * @param stdClass $request
     * @param array    $response
     * @param string   $requestAt
     * @param string   $endpoint
     * @param string   $originShop
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveTransferSync($request, $response, $requestAt, $endpoint, $originShop)
    {
        $syncedOrder = $this->syncedOrderFactory->create();

        $syncedOrder->setOrderId($this->currentJob->getEntity()->getMagentoId())
            ->setTransactionId($response['Id'])
            ->setTransactionType($this->getTransferTransactionType())
            ->setRequest($this->jsonSerializer->serialize($request))
            ->setEndpoint($endpoint)
            ->setStatus($response['status'])
            ->setRequestAt($requestAt)
            ->setResultAt(date('Y-m-d H:i:s'))
            ->setOriginShop($originShop)
            ->setTargetShop($this->currentJob->getEntity()->getFastmagShop());

        try {
            $this->syncedOrderRepository->save($syncedOrder);
        } catch (Exception $exception) {
            throw new JobException(__(
                'Unable to save transfer for order #%1 synchronization on Magento DB',
                $this->currentJob->getEntity()->getMagentoId()
            ));
        }
    }

    /**
     * Get transfer transaction type
     *
     * @return string
     */
    protected function getTransferTransactionType()
    {
        return $this->transactionType;
    }
}
