<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-02-08
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\Rule\TaxclassfamilyRepositoryInterface as TaxclassFamilyRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProductNotSyncableException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Importer;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Product\Variation as VariationEntity;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToMagento\Integration as IntegrationTrait;
use Magento\Eav\Api\AttributeSetRepositoryInterface as AttributeSetRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\ImportExport\Model\Import;
use Magento\ImportExport\Model\Import\ErrorProcessing\ProcessingErrorAggregatorInterface;
use Magento\Store\Api\StoreRepositoryInterface as StoreRepository;
use Magento\Tax\Api\TaxClassRepositoryInterface as TaxClassRepository;
use Magento\Tax\Model\Config as TaxConfig;

/**
 * Class Product
 *
 * Abstract class for Product related Integration workers
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
abstract class Product extends Worker
{
    use IntegrationTrait;

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /** @var Importer $importer */
    protected $importer;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var TaxClassRepository $taxClassRepository */
    protected $taxClassRepository;

    /** @var TaxclassFamilyRepository $taxclassFamilyRuleRepository */
    protected $taxclassFamilyRuleRepository;

    /** @var AttributeSetRepository $attributeSetRepository */
    protected $attributeSetRepository;

    /** @var StoreRepository $storeRepository */
    protected $storeRepository;

    /** @var Job $currentJob */
    protected $currentJob;

    /** @var ProductEntity $currentEntity */
    protected $currentEntity;

    /** @var array $dataToImport */
    protected $dataToImport = [];

    /** @var array $currentDefaultData */
    protected $currentDefaultData = [];

    /**
     * Save constructor
     *
     * @param Logger                   $logger
     * @param JobRepository            $jobRepository
     * @param Config                   $config
     * @param Json                     $jsonSerializer
     * @param Importer                 $importer
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param TaxClassRepository       $taxClassRepository
     * @param TaxclassFamilyRepository $taxclassFamilyRuleRepository
     * @param AttributeSetRepository   $attributeSetRepository
     * @param StoreRepository          $storeRepository
     *
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        Importer $importer,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        TaxClassRepository $taxClassRepository,
        TaxclassFamilyRepository $taxclassFamilyRuleRepository,
        AttributeSetRepository $attributeSetRepository,
        StoreRepository $storeRepository
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->config = $config;
        $this->jsonSerializer = $jsonSerializer;
        $this->importer = $importer;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->taxClassRepository = $taxClassRepository;
        $this->taxclassFamilyRuleRepository = $taxclassFamilyRuleRepository;
        $this->attributeSetRepository = $attributeSetRepository;
        $this->storeRepository = $storeRepository;

        $this->configureImporter();
    }

    /**
     * Configure importer
     *
     * @return void
     */
    protected function configureImporter()
    {
        $this->importer->setBehavior(Import::BEHAVIOR_APPEND)
            ->setEntityCode('catalog_product')
            ->setValidationStrategy(ProcessingErrorAggregatorInterface::VALIDATION_STRATEGY_SKIP_ERRORS)
            ->setAllowedErrorCount(100)
            ->setIgnoreDuplicates(false)
            ->setCategoryPathSeparator('|');
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ENABLE);
    }

    /**
     * Get job's entity ID
     *
     * @param Job $job
     *
     * @return string
     */
    protected function getJobEntityId($job)
    {
        /** @var ProductEntity $entity */
        $entity = $job->getEntity();

        return $entity->getRef();
    }

    /**
     * Check if products fill all the condition to be synced
     *
     * @param ProductEntity $entity
     *
     * @return true
     *
     * @throws ProductNotSyncableException
     */
    protected function productMustBeSynced($entity)
    {
        if (!$entity->getMagentoId()) {
            if (!($entity->getActive() && $entity->getVisibleWeb())
                && !$this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ACTIVE_VISIBLEWEB)
            ) {
                throw new ProductNotSyncableException(__(
                    'Product "%1" can\'t be synced, it\'s inactive or not in stock: %2',
                    $entity->getRef(),
                    $this->jsonSerializer->serialize($entity->export())
                ));
            }

            $onlyDefinedStocks = $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS);
            if ($onlyDefinedStocks && $entity->getStockLevel() === 0) {
                throw new ProductNotSyncableException(__(
                    'Product "%1" can\'t be synced, there\'s no inventory in defined stock in Fastmag',
                    $entity->getRef(),
                    $this->jsonSerializer->serialize($entity->export())
                ));
            }
        }

        return true;
    }

    /**
     * Generate default data for the entity
     *
     * @return void
     *
     * @throws JobException
     */
    abstract protected function generateDefaultData();

    /**
     * Returns the default attribute set code for product creation
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getDefaultAttributeSetCode()
    {
        $defaultAttributeSetId = $this->config->getValue(Config::XML_PATH_PRODUCT_IMPORT_ATTRIBUTE_SET_ID);

        try {
            $defaultAttributeSet = $this->attributeSetRepository->get($defaultAttributeSetId);

            return $defaultAttributeSet->getAttributeSetName();
        } catch (NoSuchEntityException $exception) {
            throw new JobException(__(
                'Attribute set with ID #%1 not found in Magento. Please review you config.',
                $defaultAttributeSetId
            ));
        }
    }

    /**
     * Generate data to integrate for simple products
     *
     * @return array
     *
     * @throws JobException
     */
    protected function generateSimpleProductsData()
    {
        $result = [];

        $variations = $this->currentEntity->getVariationsFlat();

        foreach ($variations as $variation) {
            $simpleProductData = $this->generateSimpleProductData($variation);

            $result[$simpleProductData['default']['sku']] = $simpleProductData;
        }

        return $result;
    }
    /**
     * Generate data to integrate for simple product
     *
     * @param VariationEntity $variation
     *
     * @return array
     *
     * @throws JobException
     */
    abstract protected function generateSimpleProductData($variation);

    /**
     * Generate simple product name
     *
     * @param VariationEntity $variation
     * @param int|null        $storeId
     *
     * @return string
     */
    protected function generateSimpleProductName($variation, $storeId = null)
    {
        $result = $this->generateConfigurableProductName();

        if ($storeId !== null) {
            $parentI18n = $this->currentEntity->getI18n($storeId);

            if ($parentI18n !== null) {
                $result = $this->generateConfigurableProductName($parentI18n);
            }
        }

        if ($result !== '') {
            $result .= ' ' . $variation->getColor() . ' ' . $variation->getSize();
        }

        return trim($result);
    }

    /**
     * Generate configurable product name
     *
     * @param ProductEntity $entity
     *
     * @return string
     */
    protected function generateConfigurableProductName($entity = null)
    {
        if ($entity === null) {
            $entity = $this->currentEntity;
        }

        return trim($entity->getDesignation() . ' ' . $entity->getDesignationBis());
    }

    /**
     * Generate simple product SKU
     *
     * @param VariationEntity $variation
     *
     * @return string
     */
    protected function generateSimpleProductSku($variation)
    {
        $result = $this->currentEntity->getRef() . '-' . $variation->getColor() . '-' . $variation->getSize();

        if ($variation->getColor() === null) {
            $result = $this->currentEntity->getRef() . '-' . $variation->getSize();
        }

        return $result;
    }

    /**
     * Get price for a simple product
     *
     * @param VariationEntity $variation
     *
     * @return float
     */
    protected function getSimpleProductPrice($variation)
    {
        $result = $variation->getIndicativePrice();

        $standardPrice = (float)$variation->getStandardPrice();
        $combinationPrice = (float)$variation->getCombinationPrice();
        $ratePrice = (float)$variation->getRatePrice();

        if ($ratePrice > 0) {
            $result = $ratePrice;
        } elseif ($standardPrice > 0) {
            $result = $standardPrice;
        } elseif ($combinationPrice > 0) {
            $result = $combinationPrice;
        }

        return $this->formatPrice($result);
    }

    /**
     * Format price, including or excluding tax
     *
     * @param float $price
     *
     * @return float
     */
    protected function formatPrice($price)
    {
        $result = $price;

        if (!$this->config->isSetFlag(TaxConfig::CONFIG_XML_PATH_PRICE_INCLUDES_TAX)) {
            $vatRate = 20.0;

            if ($this->currentEntity->getShopVatRate()) {
                $vatRate = $this->currentEntity->getShopVatRate();
            }
            if ($this->currentEntity->getVatRate()) {
                $vatRate = $this->currentEntity->getVatRate();
            }

            $result = round($price / (1 + ($vatRate / 100)), 4);
        }

        return $result;
    }

    /**
     * Get global stock level, indicated by the stock level and the token is_in_stock for each store
     *
     * @param VariationEntity $variation
     *
     * @return int
     */
    protected function getSimpleProductStockLevel($variation)
    {
        $stockLevel = ($variation->getInStock() ? $variation->getStockLevel() : 0);

        $i18ns = $variation->getI18ns();
        if ($i18ns !== null && count($i18ns) > 0) {
            foreach ($i18ns as $i18n) {
                $stockLevel += $this->getSimpleProductStockLevel($i18n);
            }
        }

        return $stockLevel;
    }

    /**
     * Create data to import special prices
     *
     * @param VariationEntity $variation
     * @param bool            $forVariation
     *
     * @return array
     */
    protected function generateSpecialPriceData($variation, $forVariation = false)
    {
        $result = [];

        if ($forVariation === false) {
            $result = $this->generateSpecialPriceData($variation, true);
        }

        if ($result === []) {
            $salePrice = ($forVariation ? $variation->getVariationSalePrice() : $variation->getSalePrice());
            $salePriceDiscount =
                ($forVariation ? $variation->getVariationSaleDiscount() : $variation->getSaleDiscount());
            $salePriceReason = ($forVariation ? $variation->getVariationSaleReason() : $variation->getSaleReason());

            if (($salePrice === null && $salePriceDiscount === null) || stripos($salePriceReason, 'VIP') !== false) {
                return [];
            }

            $specialPrice = 0;
            if ($salePrice) {
                $specialPrice = $salePrice;
            } elseif ($salePriceDiscount) {
                $specialPrice = (1 - ($salePriceDiscount / 100)) * $this->getSimpleProductPrice($variation);
            }
            $result['special_price'] = number_format($specialPrice, 4, '.', '');

            $result['special_price_from_date'] =
                ($forVariation ? $variation->getVariationSaleBeginAt() : $variation->getSaleBeginAt());
            $result['special_price_to_date'] =
                ($forVariation ? $variation->getVariationSaleEndAt() : $variation->getSaleEndAt());

            if ($result['special_price_from_date'] === null) {
                $result['special_price_from_date'] = Import::DEFAULT_EMPTY_ATTRIBUTE_VALUE_CONSTANT;
            }
            if ($result['special_price_to_date'] === null) {
                $result['special_price_to_date'] = Import::DEFAULT_EMPTY_ATTRIBUTE_VALUE_CONSTANT;
            }
        }

        return $result;
    }

    /**
     * Add required attributes values in imported data to bypass potential exceptions during import
     *
     * @param VariationEntity $variation
     *
     * @return array
     */
    protected function addRequiredAttributes($variation = null)
    {
        return [];
    }

    /**
     * Check if i18nized data of a variation are different enought to create a new line in the import array
     *
     * @param VariationEntity $variation
     * @param int             $storeId
     *
     * @return bool
     */
    protected function areVariationI18nDataDifferent($variation, $storeId)
    {
        $result = false;

        if ($this->generateSimpleProductName($variation) !== $this->generateSimpleProductName($variation, $storeId)) {
            $result = true;
        }

        $i18n = $variation->getI18n($storeId);
        if ($i18n !== null && $this->getSimpleProductPrice($variation) !== $this->getSimpleProductPrice($i18n)) {
            $result = true;
        }

        return $result;
    }

    /**
     * Check if there's no error on current data, then move them to the $dataToImport variable
     *
     * @param array $simpleProductsData
     * @param array $configurableProductData
     *
     * @return void
     */
    protected function moveDataToImport($simpleProductsData, $configurableProductData)
    {
        if (count($simpleProductsData) > 0) {
            foreach ($simpleProductsData as $row) {
                $this->dataToImport[] = $this->exportDataToImport($row['default']);

                if (array_key_exists('store', $row)) {
                    foreach ($row['store'] as $storeRow) {
                        $this->dataToImport[] = $this->exportDataToImport($storeRow);
                    }
                }
            }
        }
    }

    /**
     * Export data row in the format expected by the importer
     *
     * @return array
     */
    protected function exportDataToImport($row)
    {
        if (array_key_exists('additional_attributes', $row) && is_array($row['additional_attributes'])) {
            $additionnalAttributes = '';

            foreach ($row['additional_attributes'] as $code => $value) {
                $additionnalAttributes .= $code . '=' . $value . ',';
            }

            $row['additional_attributes'] = trim($additionnalAttributes, ',');
        }

        if (array_key_exists('configurable_variations', $row) && is_array($row['configurable_variations'])) {
            $configurableVariations = '';

            foreach ($row['configurable_variations'] as $variation) {
                foreach ($variation as $code => $value) {
                    $configurableVariations .= $code . '=' . $value . ',';
                }

                $configurableVariations = trim($configurableVariations, ',') . '|';
            }

            $row['configurable_variations'] = trim($configurableVariations, '|');
        }

        return $row;
    }

    /**
     * Run product importer
     *
     * @throws JobException
     */
    protected function runImport()
    {
        $this->logger->debug(__(
            '[%1] Data to import: %2',
            $this->getCode(),
            $this->jsonSerializer->serialize($this->dataToImport)
        )->render());

        try {
            if (count($this->dataToImport) > 0) {
                $this->importer->processImport($this->dataToImport);

                $this->logger->warning(__(
                    '[%1] Import results: %2' . "\n" . 'Import errors: %3',
                    $this->getCode(),
                    $this->importer->getLogTrace(),
                    $this->importer->getErrorMessages()
                )->render());
            }
        } catch (Exception $exception) {
            $message = $exception->getMessage() . "\n" . 'Import errors: ' . $this->importer->getErrorMessages();

            throw new JobException(__($message));
        }
    }
}
