<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-03-25
 ******************************************************************************/

namespace Fastmag\Sync\Controller\Adminhtml\Catalog\Product;

use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterfaceFactory as JobFactory;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\System\Connection\Api as FastmagApi;
use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\Catalog\Api\ProductRepositoryInterface as ProductRepository;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use stdClass;

/**
 * Class Resync
 *
 * Controller class creating an import job to sync product in Magento
 */
class Resync extends Action
{
    /** @var string */
    public const ADMIN_RESOURCE = 'Fastmag_Sync::product_resync';

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /** @var JobFactory $jobFactory */
    protected $jobFactory;

    /** @var ProductRepository $productRepository */
    protected $productRepository;

    /** @var FastmagApi $api */
    protected $api;

    /**
     * Resync constructor
     *
     * @param Context           $context
     * @param JobRepository     $jobRepository
     * @param JobFactory        $jobFactory
     * @param ProductRepository $productRepository
     * @param FastmagApi        $api
     */
    public function __construct(
        Context $context,
        JobRepository $jobRepository,
        JobFactory $jobFactory,
        ProductRepository $productRepository,
        FastmagApi $api
    ) {
        parent::__construct($context);

        $this->jobRepository = $jobRepository;
        $this->jobFactory = $jobFactory;
        $this->productRepository = $productRepository;
        $this->api = $api;
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        $productId = $this->getRequest()->getParam('id');

        $fastmagRef = $this->getFastmagRef($productId);

        if ($fastmagRef !== null) {
            $job = $this->jobFactory->create();

            $job->setContentId($fastmagRef)
                ->setJobCode('tomagento_integration_product_update');

            try {
                $this->jobRepository->save($job);

                $this->messageManager->addSuccessMessage(
                    __('Product #%1 will be resynchronized in a few moments', $productId)->render()
                );
            } catch (CouldNotSaveException $exception) {
                $this->messageManager->addErrorMessage(
                    __('Unable to create a sync job for the product: %1', $exception->getMessage())->render()
                );
            }
        } else {
            $this->messageManager->addErrorMessage(
                __('Product #%1 does not seem to exist in Fastmag, it does not have a Fastmag ID', $productId)->render()
            );
        }

        $resultRedirect = $this->resultRedirectFactory->create();
        $resultRedirect->setPath('catalog/product/edit', ['id' => $productId]);

        return $resultRedirect;
    }

    /**
     * Get product Fastmag reference
     *
     * @param string $productId
     *
     * @return string|null
     */
    protected function getFastmagRef($productId)
    {
        $result = null;

        $fasmagProductId = $this->getProductFastmagId($productId);

        if ($fasmagProductId !== null) {
            $request = $this->generateProductRequest($fasmagProductId);

            try {
                $response = $this->api->post('/boa/product/search/index.ips', $request);

                if ($response['status'] === 'KO') {
                    $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                    throw new ApiException(
                        __($message),
                        $this->api->getLastRequest(),
                        null,
                        (array_key_exists('errorCode', $response) ? $response['errorCode'] : 0)
                    );
                }

                if (array_key_exists('data', $response)) {
                    $response = $response['data'][0];

                    $result = $response['Reference'];
                }
            } catch (ApiException $exception) {
                // Do nothing
            }
        }

        return $result;
    }

    /**
     * Get product Fastmag ID
     *
     * @param int $productId
     *
     * @return int|null
     */
    protected function getProductFastmagId($productId)
    {
        $result = null;

        try {
            $product = $this->productRepository->getById($productId);
        } catch (NoSuchEntityException $exception) {
            $product = null;
        }

        if ($product !== null) {
            $fastmagProductIdAttribute
                = $product->getCustomAttribute(Constants::ATTRIBUTE_PRODUCT_FASTMAG_PRODUCT_ID_CODE);

            if ($fastmagProductIdAttribute !== null) {
                $result = $fastmagProductIdAttribute->getValue();
            }
        }

        return $result;
    }

    /**
     * Generate product search request
     *
     * @param int $fastmagGencod
     *
     * @return stdClass
     */
    protected function generateProductRequest($fastmagGencod)
    {
        return (object)['Reference' => '#' . $fastmagGencod];
    }
}
