<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-03-30
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration\Customer;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Process\Entity\ToMagento\Customer as CustomerEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Customer\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Customer\AddressFactory as AddressEntityFactory;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Address
 *
 * Hydration class used for inserting or updating customers addresses from Fastmag to Magento
 * Called from Customer hydration worker only
 */
class Address extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_customer_address';

    /** @var AddressEntityFactory $addressEntityFactory */
    protected $addressEntityFactory;

    /**
     * Address constructor.
     *
     * @param Logger               $logger
     * @param JobRepository        $jobRepository
     * @param Config               $config
     * @param Json                 $jsonSerializer
     * @param Proxy                $fastmagSql
     * @param AddressEntityFactory $addressEntityFactory
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        Proxy $fastmagSql,
        AddressEntityFactory $addressEntityFactory
    ) {
        parent::__construct($logger, $jobRepository, $config, $jsonSerializer, $fastmagSql);

        $this->addressEntityFactory = $addressEntityFactory;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $results = $this->getDataFromFastmag();
        } catch (ProcessException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }

        if (isset($results)) {
            foreach ($this->getJobs()->getItems() as $job) {
                if (array_key_exists($job->getContentId(), $results)) {
                    $addressToSync = $this->getShippingAddressToSync($job, $results[$job->getContentId()]);

                    /** @var CustomerEntity $customer */
                    $customer = $job->getEntity();
                    $customer->setShippingAddress($addressToSync);
                    $this->hydrateJob($job, $customer);
                } else {
                    $exception = new JobException(
                        __('No addresses found in Fastmag for customer #%1', $job->getContentId())
                    );

                    $this->skipJob($job, $exception);
                }
            }

            foreach ($this->getJobs()->getItems() as $job) {
                try {
                    $this->saveJob($job);
                } catch (ProcessException $exception) {
                    $this->invalidateJob($job, $exception);
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $customersIds = $this->getCustomersIds();

        try {
            $sql = 'SELECT clientadresse.Client AS customer_fastmag_id, clientadresse.AdrLivraison AS alias,
                    ventes.Vente AS fastmag_order_id, Nom AS lastname, Societe AS company, Adresse1 AS street_one,
                    Adresse2 AS street_two, CodePostal AS postcode, Ville AS city, Telephone AS phone_number,
                    (SELECT CodeIso FROM pays WHERE pays.Pays = clientadresse.Pays) as country_code
                FROM clientadresse LEFT OUTER JOIN ventes
                    ON clientadresse.AdrLivraison = ventes.AdrLivraison AND clientadresse.Client = ventes.Client
                WHERE Archiver = 0 AND TRIM(clientadresse.AdrLivraison) <> \'\'
                    AND clientadresse.Client IN (' . $this->getFastmagSqlConnection()->escape($customersIds) . ')
                ORDER BY clientadresse.Client ASC, ventes.Vente DESC';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new ProcessException(__(
                'Error when hydrating customers addresses. Message: %1. Customers IDs: %2',
                $exception->getMessage(),
                implode(', ', $customersIds)
            ));
        }

        $result = [];
        foreach ($rows as $row) {
            $addressEntity = $this->addressEntityFactory->create();
            $addressEntity->setData($row);

            if (!array_key_exists((int)$row['customer_fastmag_id'], $result)) {
                $result[(int)$row['customer_fastmag_id']] = [];
            }
            $result[(int)$row['customer_fastmag_id']][$row['alias']] = $addressEntity;
        }

        return $result;
    }

    /**
     * Returns the list of the customers IDs, if the jobs are linked to customers.
     *
     * @return int[]
     */
    protected function getCustomersIds()
    {
        return $this->getJobs()->getColumnValues(Job::CONTENT_ID);
    }

    /**
     * Select address to sync for a single customer
     *
     * @param Job             $job
     * @param AddressEntity[] $addresses
     *
     * @return AddressEntity
     */
    protected function getShippingAddressToSync($job, $addresses)
    {
        $result = [];
        $lastOrderId = null;

        foreach ($addresses as $address) {
            $this->log($job, 'Check if address "' . $address->getAlias() . '" has to be synced');

            if ($lastOrderId === null || $address->getFastmagOrderId() > $lastOrderId) {
                $result = $address;
                $lastOrderId = $address->getFastmagOrderId();
            }
        }

        $this->log($job, 'Address "' . $result->getAlias() . '" has to be synced');

        return $result;
    }
}
