<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-03-30
 ******************************************************************************/

namespace Fastmag\Sync\Plugin\Jobqueue\ToFastmag;

use Exception;
use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Jobqueue\ToFastmagFactory as JobFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\CustomerFactory as CustomerEntityFactory;
use Fastmag\Sync\Plugin\Jobqueue\ToFastmag;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Customer\Api\Data\CustomerInterface as CustomerModel;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * Class Customer
 *
 * Plugin on CustomerRepository to add a matching job in ToFastmag job queue
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Customer extends ToFastmag
{
    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var CustomerEntityFactory $customerEntityFactory */
    protected $customerEntityFactory;

    /** @var SyncedCustomerRepository $syncedCustomerRepository */
    protected $syncedCustomerRepository;

    /**
     * Customer contructor
     *
     * @param JobFactory               $jobFactory
     * @param JobRepository            $jobRepository
     * @param Logger                   $logger
     * @param Config                   $config
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param CustomerEntityFactory    $customerEntityFactory
     * @param SyncedCustomerRepository $syncedCustomerRepository
     */
    public function __construct(
        JobFactory $jobFactory,
        JobRepository $jobRepository,
        Logger $logger,
        Config $config,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CustomerEntityFactory $customerEntityFactory,
        SyncedCustomerRepository $syncedCustomerRepository
    ) {
        parent::__construct($jobFactory, $jobRepository, $logger, $config);

        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->customerEntityFactory = $customerEntityFactory;
        $this->syncedCustomerRepository = $syncedCustomerRepository;
    }

    /**
     * @see CustomerRepository::save()
     *
     * @param CustomerRepository $subject
     * @param CustomerModel      $customer
     *
     * @return CustomerModel
     *
     * @throws CouldNotDeleteException
     * @throws CouldNotSaveException
     */
    public function afterSave(CustomerRepository $subject, CustomerModel $customer)
    {
        $customerId = $customer->getId();

        if ($customerId && $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_EXPORT_ENABLE)) {
            $searchCriteria = $this->searchCriteriaBuilder->addFilter(Job::CONTENT_ID, $customerId)
                ->addFilter(Job::JOB_CODE, 'tofastmag_integration_customer_deletion')
                ->create();

            $deletionJobsList = $this->jobRepository->getList($searchCriteria);

            if ($deletionJobsList->getTotalCount() > 0) {
                foreach ($deletionJobsList->getItems() as $deletionJob) {
                    $this->jobRepository->delete($deletionJob);
                }
            }

            if (!$this->isCalledByFastmagModule()) {
                $job = $this->jobFactory->create();
                $job->setContentId($customerId)
                    ->setJobCode('tofastmag_integration_customer');

                $this->jobRepository->save($job);
            }
        }

        return $customer;
    }

    /**
     * @see CustomerRepository::deleteById()
     *
     * @param CustomerRepository $subject
     * @param int                $customerId
     *
     * @return void
     */
    public function beforeDeleteById(CustomerRepository $subject, int $customerId)
    {
        if ($customerId && $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_GDPR_DELETE_DATA_ACTION)) {
            try {
                $customer = $subject->getById($customerId);

                $this->addDeletionJob($customer);
            } catch (Exception $exception) {
                // Do nothing, since there's no customer to delete after all
            }
        }
    }

    /**
     * @see CustomerRepository::delete()
     *
     * @param CustomerRepository $subject
     * @param CustomerModel      $customer
     *
     * @return void
     */
    public function beforeDelete(CustomerRepository $subject, CustomerModel $customer)
    {
        if ($this->config->isSetFlag(Config::XML_PATH_CUSTOMER_GDPR_DELETE_DATA_ACTION)) {
            $this->addDeletionJob($customer);
        }
    }

    /**
     * Add deletion job in job queue for the deleted customer, and check if there's older integration jobs to delete
     *
     * @param CustomerModel $customer
     *
     * @return void
     */
    protected function addDeletionJob($customer)
    {
        $searchCriteria = $this->searchCriteriaBuilder->addFilter(Job::CONTENT_ID, $customer->getId())
            ->addFilter(Job::JOB_CODE, 'tofastmag_integration_customer')
            ->create();

        $savingJobsList = $this->jobRepository->getList($searchCriteria);

        if ($savingJobsList->getTotalCount() > 0) {
            foreach ($savingJobsList->getItems() as $savingJob) {
                try {
                    $this->jobRepository->delete($savingJob);
                } catch (CouldNotDeleteException $exception) {
                    $this->logger->error(__(
                        'Unable to delete customer #%1 integration job #%2 before adding deletion job',
                        $customer->getId(),
                        $savingJob->getId()
                    ));
                }
            }
        }

        $customerId = $customer->getId();

        $job = $this->jobFactory->create();
        $job->setContentId($customerId)
            ->setJobCode('tofastmag_integration_customer_deletion');

        try {
            $customerData = $this->getDeletedCustomerData($customer);

            $job->setHydratedData($customerData);

            $this->jobRepository->save($job);
        } catch (NoSuchEntityException $exception) {
            // Do nothing, as the customer does not exist in Fastmag in the first place
        } catch (Exception $exception) {
            $message = __('Unable to get deleted customer #%1 data, deletion job not created', $customerId);
            $this->logger->error($message);
        }
    }

    /**
     * Acts as hydration worker to get customer data before its deletion
     *
     * @param CustomerModel $customer
     *
     * @return array
     *
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    protected function getDeletedCustomerData($customer)
    {
        $customerId = $customer->getId();
        $syncedCustomer = $this->syncedCustomerRepository->getByMagentoId($customerId);

        $customerEntity = $this->customerEntityFactory->create();
        $customerEntity->setMagentoId($customerId)
            ->setEmailAddress($customer->getEmail())
            ->setFastmagId($syncedCustomer->getFastmagCustomerId())
            ->setStoreId($customer->getStoreId());

        return $customerEntity->export();
    }
}
