<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-03-30
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Customer\Address;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Address as AbstractAddress;
use Monolog\Logger as Monolog;

/**
 * Class Archiving
 *
 * Integration class used for archiving address from Magento to Fastmag
 */
class Archiving extends AbstractAddress
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_customer_address_archiving';

    /** @var string $hydrationWorker */
    protected $hydrationWorker = 'tofastmag_hydration_customer_address_archiving';

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_EXPORT_ENABLE);
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        $entity = $this->getCustomerEntity($job);

        $shippingAddress = $entity->getShippingAddress();

        if ($shippingAddress) {
            $shippingAddressAlias = $shippingAddress->getAlias();

            if ($this->checkIfAddressExistsInFastmag($entity->getFastmagId(), $shippingAddressAlias)) {
                $this->saveFastmagAddress($entity->getMagentoId(), $entity->getEmailAddress(), $shippingAddress);
            } else {
                $this->log(
                    $job,
                    __('[ERROR] Address "%1" does not seem to exist in Fastmag', $shippingAddressAlias)->render(),
                    Monolog::WARNING
                );
            }
        } else {
            $this->log($job, __('[ERROR] No address set in job, skipping.')->render(), Monolog::ERROR);
        }
    }

    /**
     * @inheritDoc
     */
    protected function getCustomerEntity($job)
    {
        return $job->getEntity();
    }

    /**
     * Check if address exists in Fastmag before archiving it
     *
     * @param int    $customerFastmagId
     * @param string $addressAlias
     *
     * @return bool
     */
    protected function checkIfAddressExistsInFastmag($customerFastmagId, $addressAlias)
    {
        try {
            $sql = 'SELECT COUNT(*) AS addresses_count
                FROM clientadresse
                WHERE Client = ' . $this->getFastmagSqlConnection()->escape($customerFastmagId) . '
                    AND AdrLivraison = ' . $this->getFastmagSqlConnection()->escape($addressAlias, true);

            $rows = $this->getFastmagSqlConnection()->get($sql);

            $result = ((int)$rows[0]['addresses_count'] !== 0);
        } catch (Exception $exception) {
            $result = false;
        }

        return $result;
    }
}
