<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-03-30
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Order;

use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Customer\Data as StandardCustomer;
use Monolog\Logger as Monolog;

/**
 * Class Customer
 *
 * Integration class used for inserting or updating orders customers from Magento to Fastmag
 */
class Customer extends StandardCustomer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_order_customer';

    /** @var string[] $subordinateWorkersAfter */
    protected $subordinateWorkersAfter = [];

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_ORDER_EXPORT_ENABLE);
    }

    /**
     * @inheritDoc
     */
    protected function processJob($job)
    {
        /** @var OrderEntity $orderEntity */
        $orderEntity = $job->getEntity();
        $transactionType = $orderEntity->getTransactionType();

        if ($this->useCustomerData($transactionType)) {
            if ($this->isCustomerSyncable($orderEntity)) {
                parent::processJob($job);
            } else {
                $this->log(
                    $job,
                    __('Order\'s billing address not in the customer\'s address book, it will not be synced.')->render(),
                    Monolog::INFO
                );
            }
        } else {
            $this->log(
                $job,
                __('Transaction %1 does not need customer data to be synced.', $transactionType)->render(),
                Monolog::INFO
            );
        }
    }

    /**
     * @inheritDoc
     *
     * @throws JobException
     */
    protected function getCustomerEntity($job)
    {
        /** @var OrderEntity $orderEntity */
        $orderEntity = $job->getEntity();

        $result = $orderEntity->getCustomer();
        if ($result === null) {
            throw new JobException(__('No customer for order #%1', $orderEntity->getIncrementId()));
        }

        $billingAddress = $orderEntity->getBillingAddress();
        if ($billingAddress === null) {
            throw new JobException(__('No billing address for order #%1', $orderEntity->getIncrementId()));
        }

        $result->setBillingAddress($billingAddress);

        return $result;
    }

    /**
     * Returns true if the transaction use customer data
     *
     * @param string|null $transactionType
     *
     * @return bool
     */
    protected function useCustomerData($transactionType)
    {
        return $transactionType === null || in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_ORDER,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE
            ],
            true
        );
    }

    /**
     * Check if customer is already synced and if the billing address used is the default one.
     * If not, we do not sync customer, as it will wrongly update its default billing address in M2
     * when syncing back with Fastmag
     *
     * @param OrderEntity $orderEntity
     *
     * @return bool
     */
    protected function isCustomerSyncable($orderEntity)
    {
        $result = true;

        $customer = $orderEntity->getCustomer();
        $billingAddress = $orderEntity->getBillingAddress();

        $entitiesFilled = $customer && $billingAddress;
        $customerIsSynced = (bool)$customer->getFastmagId();

        if (!$entitiesFilled || ($customerIsSynced && !$billingAddress->getIsDefault())) {
            $result = false;
        }

        return $result;
    }
}
