<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-06-03
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;

use DateTime;
use Fastmag\Sync\Api\Data\Rule\TaxclassfamilyInterface as TaxclassFamilyRuleModel;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\Rule\TaxclassFamilyRepositoryInterface as TaxclassFamilyRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Exception\ProductNotSyncableException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\Importer;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Product\Variation as VariationEntity;
use Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;
use Magento\Catalog\Api\CategoryListInterface as CategoryList;
use Magento\Catalog\Api\CategoryRepositoryInterface as CategoryRepository;
use Magento\Catalog\Api\Data\CategoryInterface as CategoryModel;
use Magento\Catalog\Api\Data\CategoryInterfaceFactory as CategoryFactory;
use Magento\Catalog\Api\Data\ProductInterface as ProductModel;
use Magento\Catalog\Api\Data\ProductAttributeInterface as ProductAttribute;
use Magento\Catalog\Api\ProductAttributeRepositoryInterface as AttributeRepository;
use Magento\Catalog\Api\ProductRepositoryInterface as ProductRepository;
use Magento\Catalog\Model\Product\Type;
use Magento\Catalog\Model\ResourceModel\Category\Tree;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Eav\Api\AttributeSetRepositoryInterface as AttributeSetRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\ImportExport\Model\Import;
use Magento\InventoryReservationsApi\Model\GetReservationsQuantityInterface as ReservationsQuantityGetter;
use Magento\InventorySalesApi\Api\StockResolverInterface as StockResolver;
use Magento\Store\Api\StoreRepositoryInterface as StoreRepository;
use Magento\Store\Api\WebsiteRepositoryInterface as WebsiteRepository;
use Magento\Tax\Api\TaxClassRepositoryInterface as TaxClassRepository;
use Magento\Tax\Helper\Data as TaxHelper;
use Monolog\Logger as Monolog;

/**
 * Class Save
 *
 * Integration worker for tomagento_integration_product_create and tomagento_integration_product_update jobs
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Data extends Product
{
    /** @var string */
    public const PROCESS_CATEGORY_NAME = 'Produits à traiter';

    /** @inheritDoc */
    protected $code = 'tomagento_integration_product_save';

    /** @var AttributeSetRepository $attributeSetRepository */
    protected $attributeSetRepository;

    /** @var CategoryRepository $categoryRepository */
    protected $categoryRepository;

    /** @var CategoryFactory $categoryFactory */
    protected $categoryFactory;

    /** @var CategoryList $categoryList */
    protected $categoryList;

    /** @var Tree $tree */
    protected $tree;

    /** @var ProductRepository $productRepository */
    protected $productRepository;

    /** @var AttributeRepository $attributeRepository */
    protected $attributeRepository;

    /** @var string $hydrationWorker */
    protected $hydrationWorker = 'tomagento_hydration_product';

    /** @var string[] $subordinateWorkersBefore */
    protected $subordinateWorkersBefore = ['tomagento_integration_product_option'];

    /** @var array $currentVariations */
    protected $currentVariations = [];

    /** @var ProductAttribute $colorAttribute */
    protected $colorAttribute;

    /** @var ProductAttribute $internalColorAttribute */
    protected $internalColorAttribute;

    /** @var ProductAttribute $sizeAttribute */
    protected $sizeAttribute;

    /** @var string $processCategoryPath */
    protected $processCategoryPath;

    /** @var CategoryModel $processCategory */
    protected $processCategory;

    /** @var string $defaultAttributeSetCode */
    protected $defaultAttributeSetCode;

    /**
     * Save constructor
     *
     * @param Logger                     $logger
     * @param JobRepository              $jobRepository
     * @param Config                     $config
     * @param Json                       $jsonSerializer
     * @param Importer                   $importer
     * @param SearchCriteriaBuilder      $searchCriteriaBuilder
     * @param TaxClassRepository         $taxClassRepository
     * @param TaxclassFamilyRepository   $taxclassFamilyRuleRepository
     * @param StoreRepository            $storeRepository
     * @param WebsiteRepository          $websiteRepository
     * @param ReservationsQuantityGetter $reservationsQuantityGetter
     * @param StockResolver              $stockResolver
     * @param AttributeSetRepository     $attributeSetRepository
     * @param CategoryRepository         $categoryRepository
     * @param CategoryFactory            $categoryFactory
     * @param CategoryList               $categoryList
     * @param Tree                       $tree
     * @param ProductRepository          $productRepository
     * @param AttributeRepository        $attributeRepository
     *
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        Importer $importer,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        TaxClassRepository $taxClassRepository,
        TaxclassFamilyRepository $taxclassFamilyRuleRepository,
        StoreRepository $storeRepository,
        WebsiteRepository $websiteRepository,
        ReservationsQuantityGetter $reservationsQuantityGetter,
        StockResolver $stockResolver,
        AttributeSetRepository $attributeSetRepository,
        CategoryRepository $categoryRepository,
        CategoryFactory $categoryFactory,
        CategoryList $categoryList,
        Tree $tree,
        ProductRepository $productRepository,
        AttributeRepository $attributeRepository
    ) {
        parent::__construct(
            $logger,
            $jobRepository,
            $config,
            $jsonSerializer,
            $importer,
            $searchCriteriaBuilder,
            $taxClassRepository,
            $taxclassFamilyRuleRepository,
            $storeRepository,
            $websiteRepository,
            $reservationsQuantityGetter,
            $stockResolver
        );

        $this->attributeSetRepository = $attributeSetRepository;
        $this->categoryRepository = $categoryRepository;
        $this->categoryFactory = $categoryFactory;
        $this->categoryList = $categoryList;
        $this->tree = $tree;
        $this->productRepository = $productRepository;
        $this->attributeRepository = $attributeRepository;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->error('[' . $this->getCode() . '] ' . $exception->getMessage());

            return;
        }

        try {
            $this->getColorSizeAttributesCode();
            $this->checkProcessCategory();
        } catch (ProcessException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }

        foreach ($this->getJobs()->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                } catch (ProductNotSyncableException $exception) {
                    $this->skipJob($job, $exception);
                }
            } else {
                $this->log($job, 'Job previously in error, skip integration.', Monolog::NOTICE);
            }
        }

        try {
            $this->runImport();
        } catch (JobException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }

        foreach ($this->getJobs()->getItems() as $job) {
            $this->saveJob($job);
        }
    }

    /**
     * @inheritDoc
     */
    protected function initiate()
    {
        parent::initiate();

        $defaultAttributeSetId = $this->config->getValue(Config::XML_PATH_PRODUCT_IMPORT_ATTRIBUTE_SET_ID);

        if ($defaultAttributeSetId === null) {
            throw new ProcessException(__(
                'No default attribute set defined in configuration. Please fill the config field before enabling product sync.'
            ));
        }

        try {
            $defaultAttributeSet = $this->attributeSetRepository->get($defaultAttributeSetId);

            $this->defaultAttributeSetCode = $defaultAttributeSet->getAttributeSetName();
        } catch (NoSuchEntityException $exception) {
            throw new ProcessException(__(
                'Attribute set with ID #%1 not found in Magento. Please review you config.',
                $defaultAttributeSetId
            ));
        }
    }

    /**
     * Get the color/size attributes from config and set their code for the jobs
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getColorSizeAttributesCode()
    {
        $colorAttributeId = $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_COLOR);
        $this->colorAttribute = $this->getAttribute($colorAttributeId, 'color');

        $internalColorAttributeId =
            $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_INTERNAL_COLOR);
        $this->internalColorAttribute = $this->getAttribute($internalColorAttributeId, 'internal color');

        $sizeAttributeId = $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_SIZE);
        $this->sizeAttribute = $this->getAttribute($sizeAttributeId, 'size');
    }

    /**
     * Get attribute
     *
     * @param int    $attributeId
     * @param string $label
     *
     * @return ProductAttribute
     *
     * @throws ProcessException
     */
    protected function getAttribute($attributeId, $label)
    {
        try {
            return $this->attributeRepository->get($attributeId);
        } catch (NoSuchEntityException $exception) {
            throw new ProcessException(
                __('Attribute "%1", used to save "%2" data, does not exist in Magento', $attributeId, $label)
            );
        }
    }

    /**
     * Get Fastmag process category ID, and create it if it does not exist yet
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function checkProcessCategory()
    {
        $this->searchCriteriaBuilder->addFilter(CategoryModel::KEY_NAME, self::PROCESS_CATEGORY_NAME);

        $searchCriteria = $this->searchCriteriaBuilder->create();

        $categories = $this->categoryList->getList($searchCriteria);

        if ($categories->getTotalCount() === 0) {
            $category = $this->categoryFactory->create();

            $category->setName(self::PROCESS_CATEGORY_NAME)
                ->setIncludeInMenu(false)
                ->setIsActive(false)
                ->setParentId(1);

            try {
                $category = $this->categoryRepository->save($category);
            } catch (CouldNotSaveException $exception) {
                throw new ProcessException(__($exception->getMessage()));
            }
        } else {
            $category = $categories->getItems()[0];
        }

        $this->processCategory = $category;
    }

    /**
     * Get full path of the process category
     *
     * @return string
     */
    protected function getProcessCategoryFullPath()
    {
        if ($this->processCategoryPath === null) {
            $categoryTree = $this->tree->loadBreadcrumbsArray($this->processCategory->getPath());

            $categoryTreePath = [];
            foreach ($categoryTree as $eachCategory) {
                $categoryTreePath[] = $eachCategory['name'];
            }

            $this->processCategoryPath = implode('/', $categoryTreePath);
        }

        return $this->processCategoryPath;
    }

    /**
     * @inheritDoc
     *
     * @throws ProductNotSyncableException
     */
    protected function processJob($job)
    {
        $this->currentJob = $job;
        $this->currentVariations = [];
        $this->currentDefaultData = [];

        /** @var ProductEntity $entity */
        $entity = $job->getEntity();
        $this->currentEntity = $entity;

        if ($this->productMustBeSynced($entity)) {
            $this->generateDefaultData();
            $simpleProductsData = $this->generateSimpleProductsData();
            $configurableProductData = $this->generateConfigurableProductData();

            $this->moveDataToImport($simpleProductsData, $configurableProductData);
        }
    }

    /**
     * Check if products fill all the condition to be synced
     *
     * @param ProductEntity $entity
     *
     * @return true
     *
     * @throws ProductNotSyncableException
     */
    protected function productMustBeSynced($entity)
    {
        parent::productMustBeSynced($entity);

        $this->checkRequiredValues($entity);

        return true;
    }

    /**
     * Check if products fill all the condition to be synced
     *
     * @param ProductEntity $entity
     *
     * @return void
     *
     * @throws ProductNotSyncableException
     */
    protected function checkRequiredValues($entity)
    {
        if (trim($entity->getDesignation()) === '' && trim($entity->getDesignationBis()) === '') {
            throw new ProductNotSyncableException(__(
                'Product "%1" can\'t be synced, there\'s no designation set in Fastmag',
                $entity->getRef(),
                $this->jsonSerializer->serialize($entity->export())
            ));
        }
    }

    /**
     * @inheritDoc
     */
    protected function generateDefaultData()
    {
        $this->currentDefaultData = [
            'attribute_set_code' => $this->defaultAttributeSetCode,
            'product_websites'   => implode(',', $this->getWebsitesCodes()),
            'tax_class_name'     => $this->getTaxClassName()
        ];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_CREATION_DATE)) {
            $this->currentDefaultData['created_at'] = $this->currentEntity->getCreationDate();
        }
    }

    /**
     * Get default tax class name for products
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getTaxClassName()
    {
        $taxClassId = $this->config->getValue(TaxHelper::CONFIG_DEFAULT_PRODUCT_TAX_CLASS);

        if ($this->currentEntity->getVatFamily()) {
            $this->searchCriteriaBuilder->addFilter(
                TaxclassFamilyRuleModel::FAMILY,
                $this->currentEntity->getVatFamily()
            );

            $searchCriteria = $this->searchCriteriaBuilder->create();

            $taxclassFamilyRulesList = $this->taxclassFamilyRuleRepository->getList($searchCriteria);

            if ($taxclassFamilyRulesList->getTotalCount() > 0) {
                $taxclassFamilyRuleListItems = $taxclassFamilyRulesList->getItems();
                $taxclassFamilyRule = reset($taxclassFamilyRuleListItems);
                $taxClassId = $taxclassFamilyRule->getTaxClassId();
            } else {
                $this->log(
                    $this->currentJob,
                    'Product has a family with specific VAT rate in Fastmag,'
                        . ' but there\'s no equivalent tax class in Magento',
                    Monolog::WARNING
                );
            }
        }

        try {
            $taxClass = $this->taxClassRepository->get($taxClassId);

            return $taxClass->getClassName();
        } catch (NoSuchEntityException $exception) {
            throw new JobException(__('Tax class ID #%1 does not exist in Magento', $taxClassId));
        }
    }

    /**
     * Get default Fastmag attributes as an array
     *
     * @param ProductEntity $entity
     *
     * @return array
     */
    protected function getDefaultAdditionalAttributes($entity)
    {
        $result = [
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_BRAND_CODE    => $entity->getBrand(),
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_SUPPLIER_CODE => $entity->getSupplier()
        ];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_FAMILY)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_FAMILY_CODE] = $entity->getFamily();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SUBFAMILY)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_SUBFAMILY_CODE] = $entity->getSubfamily();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SECTION)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_SECTION_CODE] = $entity->getSection();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SEASON)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_SEASON_CODE] = $entity->getSeason();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MODEL)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_MODEL_CODE] = $entity->getModel();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MATERIAL)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_MATERIAL_CODE] = $entity->getMaterial();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_THEME)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_THEME_CODE] = $entity->getTheme();
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_TEXTILE)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_WASHING_CODE] = $entity->getWashing();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_CHLORINE_CODE] = $entity->getChlorine();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_IRONING_CODE] = $entity->getIroning();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_DRY_CLEANING_CODE] = $entity->getDryCleaning();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_DRYING_CODE] = $entity->getDrying();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_WATER_CODE] = $entity->getWater();
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    protected function generateSimpleProductData($variation)
    {
        $result = [];

        $defaultData = [
            'sku'          => $this->generateSimpleProductSku($variation),
            'product_type' => Type::TYPE_SIMPLE,
            'name'         => $this->generateSimpleProductName($variation),
            'weight'       => number_format($variation->getWeight(), 4, '.', ''),
            'visibility'   => __('Not Visible Individually'),
            'price'        => number_format($this->getSimpleProductPrice($variation), 4, '.', ''),
            'cost'         => number_format($variation->getBuyingPrice(), 4, '.', ''),
            'url_key'      => $this->generateUrlKey($this->generateSimpleProductName($variation), 'simple'),
            'qty'          => $this->getSimpleProductStockLevel($variation),
            'additional_attributes' => $this->getSimpleProductAdditionalAttributes($variation)
        ];

        try {
            $product = $this->productRepository->get($defaultData['sku']);

            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_MANAGE_PRODUCT_STATUS)) {
                $defaultData['product_online'] = $this->getProductStatus($variation, $product);
            }
        } catch (NoSuchEntityException $exception) {
            $defaultData['product_online'] = '0';
            $defaultData['categories'] = $this->getProcessCategoryFullPath();
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_IS_NEW)) {
            if ($this->currentEntity->getIsNew()) {
                $newFromDate = new DateTime();

                $defaultData['news_from_date'] = $newFromDate->format('Y-m-d H:i:s');
            } else {
                $defaultData['news_from_date'] = Import::DEFAULT_EMPTY_ATTRIBUTE_VALUE_CONSTANT;
            }
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRICE_IMPORT_SPECIAL_PRICES)) {
            $defaultData = array_merge($defaultData, $this->generateSpecialPriceData($variation));
        }

        $result['default'] = array_merge(
            $this->currentDefaultData,
            $defaultData,
            $this->addRequiredAttributes($variation)
        );

        $this->currentVariations[] = $this->addToCurrentVariations(
            $defaultData['sku'],
            $defaultData['additional_attributes'][$this->colorAttribute->getAttributeCode()],
            $defaultData['additional_attributes'][$this->sizeAttribute->getAttributeCode()]
        );

        $i18nData = $this->generateSimpleProductI18nData($variation);
        if (count($i18nData) > 0) {
            $result['store'] = $i18nData;
        }

        $this->log(
            $this->currentJob,
            'Data to import for simple product "' . $defaultData['sku'] . '": '
            . $this->jsonSerializer->serialize($result)
        );

        return $result;
    }

    /**
     * Create i18nized data to import for simple products
     *
     * @param VariationEntity $variation
     *
     * @return array
     *
     * @throws JobException
     */
    protected function generateSimpleProductI18nData($variation)
    {
        $result = [];

        $i18ns = $variation->getI18ns();

        if ($i18ns !== null && count($i18ns) > 0) {
            foreach ($i18ns as $storeId => $i18n) {
                if ($this->areVariationI18nDataDifferent($variation, $storeId)) {
                    $i18nData = [
                        'sku'                => $this->generateSimpleProductSku($variation),
                        'store_view_code'    => $this->getStoreViewCode($storeId),
                        'attribute_set_code' => $this->defaultAttributeSetCode,
                        'product_type'       => Type::TYPE_SIMPLE,
                        'name'               => $this->generateSimpleProductName($variation, $storeId),
                        'price'              => $this->getSimpleProductPrice($i18n),
                        'url_key'            => $this->generateUrlKey(
                            $this->generateSimpleProductName($variation, $storeId),
                            'simple'
                        ),
                        // Delete all default values unnecessary for store override
                        'product_websites'   => null,
                        'tax_class_name'     => null
                    ];

                    $i18nData = array_filter($i18nData, static function ($value) {
                        if (is_array($value)) {
                            return array_filter($value);
                        }

                        return $value !== null && trim($value) !== '';
                    });

                    $result[$i18nData['store_view_code']] = array_merge($this->currentDefaultData, $i18nData);
                }
            }
        }

        return $result;
    }

    /**
     * Generate configurable product SKU
     *
     * @return string
     */
    protected function generateConfigurableProductSku()
    {
        return $this->currentEntity->getRef();
    }

    /**
     * Generate product URL key
     *
     * @param string $productName
     * @param string $type
     *
     * @return string
     */
    protected function generateUrlKey($productName, $type = 'configurable')
    {
        $result = strtolower(preg_replace('#[^0-9a-z]+#i', '-', $productName));

        if ($type !== 'configurable' && $productName === $this->generateConfigurableProductName()) {
            $result .= '-nr';
        }

        return $result;
    }

    /**
     * Get Fastmag attributes values for a simple product
     *
     * @param VariationEntity $variation
     *
     * @return array
     */
    protected function getSimpleProductAdditionalAttributes($variation)
    {
        $result = $this->getDefaultAdditionalAttributes($this->currentEntity);

        $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_PRODUCT_ID_CODE] = $variation->getFastmagId();
        $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_EAN13_CODE] = $variation->getEan13();

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_USE_COLOR_DESCRIPTION)) {
            $result[$this->colorAttribute->getAttributeCode()] = $variation->getColorDescription();
            $result[$this->internalColorAttribute->getAttributeCode()] = $variation->getColor();
        } else {
            $result[$this->colorAttribute->getAttributeCode()] = $variation->getColor();
        }

        $result[$this->sizeAttribute->getAttributeCode()] = $variation->getSize();
        if ($variation->getSize() === ''
            && $this->config->isSetFlag(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_UNIQUE_SIZE)
        ) {
            $result[$this->sizeAttribute->getAttributeCode()] =
                $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_UNIQUE_SIZE_LABEL);
        }

        return $result;
    }

    /**
     * Get product enability
     *
     * @param VariationEntity|null $variation
     * @param ProductModel|null    $product
     *
     * @return string
     */
    protected function getProductStatus($variation = null, $product = null)
    {
        $result = $this->currentEntity->getActive() && $this->currentEntity->getVisibleWeb();

        if ($result && $variation !== null) {
            $result = $variation->getActive();
        }

        if ($product === null) {
            $sku = $this->generateConfigurableProductSku();
            if ($variation) {
                $sku = $this->generateSimpleProductSku($variation);
            }

            try {
                $product = $this->productRepository->get($sku);
            } catch (NoSuchEntityException $exception) {
                return 0;
            }
        }

        $categoryIds = $product->getCategoryIds();

        if (in_array($this->processCategory->getId(), $categoryIds, true)) {
            $result = false;
        }

        return $result ? '1' : '0';
    }

    /**
     * Add to current variations list, which will be added to the configurable attribute
     *
     * @param string $sku
     * @param string $color
     * @param string $size
     *
     * @return array
     */
    protected function addToCurrentVariations($sku, $color, $size)
    {
        $result = [
            'sku' => $sku
        ];

        if (!empty($color)) {
            $result[$this->colorAttribute->getAttributeCode()] = $color;
        }
        if (!empty($size)) {
            $result[$this->sizeAttribute->getAttributeCode()] = $size;
        }

        return $result;
    }

    /**
     * Get store code by store ID
     *
     * @param int $storeId
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getStoreViewCode($storeId)
    {
        try {
            $storeView = $this->storeRepository->getById($storeId);
        } catch (NoSuchEntityException $e) {
            throw new JobException(__('Store #%1 has i18ns set in entity, but does not exist in Magento', $storeId));
        }

        return $storeView->getCode();
    }

    /**
     * Generate data to integrate for configurable product
     *
     * @return array
     *
     * @throws JobException
     */
    protected function generateConfigurableProductData()
    {
        $defaultData = [
            'sku'          => $this->generateConfigurableProductSku(),
            'product_type' => Configurable::TYPE_CODE,
            'name'         => $this->generateConfigurableProductName(),
            'weight'       => number_format($this->currentEntity->getWeight(), 4, '.', ''),
            'visibility'   => __('Catalog, Search'),
            'url_key'      => $this->generateUrlKey($this->generateConfigurableProductName()),
            'additional_attributes'   => $this->getDefaultAdditionalAttributes($this->currentEntity),
            'configurable_variations' => $this->currentVariations
        ];

        if (!$this->currentEntity->getMagentoId()) {
            $defaultData['product_online'] = '0';
            $defaultData['categories'] = $this->getProcessCategoryFullPath();
        } elseif ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_MANAGE_PRODUCT_STATUS)) {
            $defaultData['product_online'] = $this->getProductStatus();
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_DESCRIPTION)) {
            $defaultData['description'] = $this->currentEntity->getDescription();
            $defaultData['short_description'] = $this->currentEntity->getDescription();
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_IS_NEW)) {
            if ($this->currentEntity->getIsNew()) {
                $newFromDate = new DateTime();

                $defaultData['news_from_date'] = $newFromDate->format('Y-m-d H:i:s');
            } else {
                $defaultData['news_from_date'] = Import::DEFAULT_EMPTY_ATTRIBUTE_VALUE_CONSTANT;
            }
        }

        $result['default'] = array_merge(
            $this->currentDefaultData,
            $defaultData,
            $this->addRequiredAttributes()
        );

        $i18nData = $this->generateConfigurableProductI18nData();
        if (count($i18nData) > 0) {
            $result['store'] = $i18nData;
        }

        $this->log(
            $this->currentJob,
            'Data to import for configurable product "' . $result['default']['sku'] . '": '
            . $this->jsonSerializer->serialize($result)
        );

        return $result;
    }

    /**
     * Create i18nized data to import for the current configurable product
     *
     * @return array
     *
     * @throws JobException
     */
    protected function generateConfigurableProductI18nData()
    {
        $result = [];

        $i18ns = $this->currentEntity->getI18ns();

        if ($i18ns !== null && count($i18ns) > 0) {
            foreach ($i18ns as $storeId => $i18n) {
                if ($this->areProductI18nDataDifferent($storeId)) {
                    $i18nData = [
                        'sku'                => $this->generateConfigurableProductSku(),
                        'store_view_code'    => $this->getStoreViewCode($storeId),
                        'attribute_set_code' => $this->defaultAttributeSetCode,
                        'product_type'       => Configurable::TYPE_CODE,
                        'name'               => $this->generateConfigurableProductName($i18n),
                        'url_key'            => $this->generateUrlKey($this->generateConfigurableProductName($i18n)),
                        // Delete all default values unnecessary for store override
                        'product_websites'   => null,
                        'tax_class_name'     => null
                    ];

                    if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_DESCRIPTION)) {
                        $i18nData['description'] = $i18n->getDescription();
                        $i18nData['short_description'] = $i18n->getDescription();
                    }

                    $result[$i18nData['store_view_code']] = array_merge($this->currentDefaultData, $i18nData);
                }
            }
        }

        return $result;
    }

    /**
     * Check if i18nized data of the current product are different enought to create a new line in the import array
     *
     * @param int $storeId
     *
     * @return bool
     */
    protected function areProductI18nDataDifferent($storeId)
    {
        $result = false;

        $i18n = $this->currentEntity->getI18n($storeId);

        if ($i18n !== null) {
            $i18nName = $this->generateConfigurableProductName($i18n);
            if ($i18nName !== '' && $this->generateConfigurableProductName() !== $i18nName) {
                $result = true;
            }

            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_DESCRIPTION)
                && $this->currentEntity->isI18nDifferent('description', $storeId)
            ) {
                $result = true;
            }
        }

        return $result;
    }

    /**
     * Check if there's no error on current data, then move them to the $dataToImport variable
     *
     * @param array $simpleProductsData
     * @param array $configurableProductData
     *
     * @return void
     */
    protected function moveDataToImport($simpleProductsData, $configurableProductData)
    {
        if (count($simpleProductsData) > 0) {
            if (!$this->config->isSetFlag(Config::XML_PATH_PRODUCT_CATALOG_UNIQUE_COMBINATION)
                && count($simpleProductsData) === 1
            ) {
                $dataMerged = $this->mergeConfigurableSimpleData($simpleProductsData, $configurableProductData);

                foreach ($dataMerged as $row) {
                    $this->dataToImport[] = $this->exportDataToImport($row);
                }
            } else {
                foreach ($simpleProductsData as $row) {
                    $this->dataToImport[] = $this->exportDataToImport($row['default']);

                    if (array_key_exists('store', $row)) {
                        foreach ($row['store'] as $storeRow) {
                            $this->dataToImport[] = $this->exportDataToImport($storeRow);
                        }
                    }
                }

                $this->dataToImport[] = $this->exportDataToImport($configurableProductData['default']);

                if (array_key_exists('store', $configurableProductData)) {
                    foreach ($configurableProductData['store'] as $storeRow) {
                        $this->dataToImport[] = $this->exportDataToImport($storeRow);
                    }
                }
            }
        }
    }

    /**
     * Merge parent data in variation data
     * Only used when we don't or can't create a configurable product (config set or missing configurable attributes)
     *
     * @param array $simpleProductsData
     * @param array $configurableProductData
     *
     * @return array
     */
    protected function mergeConfigurableSimpleData($simpleProductsData, $configurableProductData)
    {
        $result = [];

        $simpleProductData = reset($simpleProductsData);

        $result[] = $simpleProductData['default'];

        $result[0]['visibility'] = __('Catalog, Search');

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_DESCRIPTION)) {
            $result[0]['description'] = $configurableProductData['default']['description'];
            $result[0]['short_description'] = $configurableProductData['default']['short_description'];
        }

        if (array_key_exists('store', $simpleProductData)) {
            foreach ($simpleProductData['store'] as $storeId => $storeRow) {
                $storeRow['description'] = $configurableProductData['store'][$storeId]['description'];
                $storeRow['short_description'] = $configurableProductData['store'][$storeId]['short_description'];
                $result[] = $storeRow;
            }
        }

        return $result;
    }
}
