<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-11-18
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;

use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Exception\ProductNotSyncableException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Product\Variation as VariationEntity;
use Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;
use Magento\Framework\Exception\NoSuchEntityException;
use Monolog\Logger as Monolog;

/**
 * Class Price
 *
 * Integration worker for tomagento_integration_product_special_price jobs
 */
class Price extends Product
{
    /** @inheritDoc */
    protected $code = 'tomagento_integration_product_price';

    /** @var string $hydrationWorker */
    protected $hydrationWorker = 'tomagento_hydration_product_price';

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->error('[' . $this->getCode() . '] ' . $exception->getMessage());
            return;
        }

        foreach ($this->getJobs()->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                } catch (ProductNotSyncableException $exception) {
                    $this->skipJob($job, $exception);
                }
            } else {
                $this->log($job, 'Job previously in error, skip integration.', Monolog::NOTICE);
            }
        }

        try {
            $this->runImport();
        } catch (JobException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }

        foreach ($this->getJobs()->getItems() as $job) {
            $this->saveJob($job);
        }
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return !$this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_INVENTORY)
            && $this->config->isSetFlag(Config::XML_PATH_PRICE_IMPORT_SPECIAL_PRICES)
            && parent::isEnabled();
    }

    /**
     * @inheritDoc
     *
     * @throws ProductNotSyncableException
     */
    protected function processJob($job)
    {
        $this->currentJob = $job;
        $this->currentDefaultData = [];

        /** @var ProductEntity $entity */
        $entity = $job->getEntity();
        $this->currentEntity = $entity;

        if ($this->productMustBeSynced($entity)) {
            $this->generateDefaultData();
            $simpleProductsData = $this->generateSimpleProductsData();
        } else {
            throw new ProductNotSyncableException(__(
                'Product "%1" can\'t be synced, it must be inactive or not in stock: %2',
                $entity->getRef(),
                $this->jsonSerializer->serialize($entity->export())
            ));
        }

        $this->moveDataToImport($simpleProductsData, []);
    }

    /**
     * @inheritDoc
     */
    protected function generateDefaultData()
    {
        $this->currentDefaultData = [];
    }

    /**
     * @inheritDoc
     */
    protected function generateSimpleProductData($variation)
    {
        $result = [];

        $defaultData = [
            'sku'   => $this->generateSimpleProductSku($variation),
            'price' => number_format($this->getSimpleProductPrice($variation), 4, '.', ''),
            'cost'  => number_format($variation->getBuyingPrice(), 4, '.', '')
        ];

        if ($this->config->isSetFlag(Config::XML_PATH_PRICE_IMPORT_SPECIAL_PRICES)) {
            $defaultData = array_merge($defaultData, $this->generateSpecialPriceData($variation));
        }

        $result['default'] = array_merge(
            $this->currentDefaultData,
            $defaultData,
            $this->addRequiredAttributes($variation)
        );

        $i18nData = $this->generateSimpleProductI18nData($variation);
        if (count($i18nData) > 0) {
            $result['store'] = $i18nData;
        }

        $this->log(
            $this->currentJob,
            'Data to import for simple product "' . $defaultData['sku'] . '": '
            . $this->jsonSerializer->serialize($result)
        );

        return $result;
    }

    /**
     * Create i18nized data to import for simple products
     *
     * @param VariationEntity $variation
     *
     * @return array
     *
     * @throws JobException
     */
    protected function generateSimpleProductI18nData($variation)
    {
        $result = [];

        $i18ns = $variation->getI18ns();

        if ($i18ns !== null && count($i18ns) > 0) {
            foreach ($i18ns as $storeId => $i18n) {
                if ($this->areVariationI18nDataDifferent($variation, $storeId)) {
                    $i18nData = [
                        'sku'             => $this->generateSimpleProductSku($variation),
                        'store_view_code' => $this->getStoreViewCode($storeId),
                        'price'           => $this->getSimpleProductPrice($i18n),
                    ];

                    $i18nData = array_filter($i18nData, static function ($value) {
                        if (is_array($value)) {
                            return array_filter($value);
                        }

                        return $value !== null && trim($value) !== '';
                    });

                    $result[$i18nData['store_view_code']] = array_merge($this->currentDefaultData, $i18nData);
                }
            }
        }

        return $result;
    }

    /**
     * Get store code by store ID
     *
     * @param int $storeId
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getStoreViewCode($storeId)
    {
        try {
            $storeView = $this->storeRepository->getById($storeId);
        } catch (NoSuchEntityException $e) {
            throw new JobException(__('Store #%1 has i18ns set in entity, but does not exist in Magento', $storeId));
        }

        return $storeView->getCode();
    }
}
