<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-03-30
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration;

use Exception;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Process\Entity\ToFastmag\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\AddressFactory as AddressEntityFactory;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration as HydrationTrait;
use Magento\Customer\Api\AddressRepositoryInterface as AddressRepository;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Customer\Api\Data\AddressInterface as AddressModel;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Sales\Api\Data\OrderAddressInterface as OrderAddressModel;
use Magento\Sales\Api\Data\OrderInterface as OrderModel;
use Magento\Sales\Model\Order\Address as OrderAddress;

/**
 * Class Address
 *
 * Hydration class used for inserting or updating customers addresses from Magento to Fastmag
 */
abstract class Address extends Worker
{
    use HydrationTrait;

    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_address';

    /** @var CustomerRepository $customerRepository */
    protected $customerRepository;

    /** @var AddressRepository $addressRepository */
    protected $addressRepository;

    /** @var AddressEntityFactory $addressEntityFactory */
    protected $addressEntityFactory;

    /**
     * Address constructor.
     *
     * @param Logger               $logger
     * @param JobRepository        $jobRepository
     * @param Config                   $config
     * @param Json                     $jsonSerializer
     * @param CustomerRepository   $customerRepository
     * @param AddressRepository    $addressRepository
     * @param AddressEntityFactory $addressEntityFactory
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        CustomerRepository $customerRepository,
        AddressRepository $addressRepository,
        AddressEntityFactory $addressEntityFactory
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->config = $config;
        $this->jsonSerializer = $jsonSerializer;
        $this->customerRepository = $customerRepository;
        $this->addressRepository = $addressRepository;
        $this->addressEntityFactory = $addressEntityFactory;
    }

    /**
     * Generate an address entity from a customer or an order address
     *
     * @param AddressModel|OrderAddress $address
     * @param string                    $alias
     * @param bool                      $archive
     * @param bool                      $isDefault
     *
     * @return AddressEntity
     */
    protected function generateAddressEntity($address, $alias, $archive = false, $isDefault = true)
    {
        $result = $this->addressEntityFactory->create();

        $result->setAlias($alias)
            ->setMagentoId($address->getId())
            ->setFirstname($address->getFirstname())
            ->setLastname($address->getLastname())
            ->setCompany($address->getCompany())
            ->setStreet($address->getStreet())
            ->setPostcode($address->getPostcode())
            ->setCity($address->getCity())
            ->setCountryId($address->getCountryId())
            ->setPhoneNumber($address->getTelephone())
            ->setArchive($archive)
            ->setIsDefault($isDefault);

        return $result;
    }

    /**
     * Defines the address alias
     *
     * @param AddressModel|OrderAddressModel $address
     * @param OrderModel|null                $order
     * @param string                         $type
     *
     * @return string
     */
    protected function getAddressAlias($address, $order = null, $type = 'billing')
    {
        $result = '';

        if ($order && $order->getEntityId() && $address->getCustomerAddressId() !== null) {
            $customerAddressId = $address->getCustomerAddressId();

            try {
                $customerAddress = $this->addressRepository->getById($customerAddressId);
                $address = $customerAddress;
            } catch (Exception $exception) {
                // Do nothing
            }
        }

        $addressAliasAttribute = $address->getCustomAttribute(Constants::ATTRIBUTE_ADDRESS_ALIAS_CODE);
        if ($addressAliasAttribute !== null) {
            $result = $addressAliasAttribute->getValue();
        }

        if ($result === '.' || empty($result)) {
            if ($order !== null) {
                if ($type === 'billing') {
                    $result = 'ORD_' . $order->getEntityId() . '_INV';
                } elseif ($type === 'shipping') {
                    $result = 'ORD_' . $order->getEntityId() . '_SHIP';
                }
            } else {
                $result = 'ADDR_' . $address->getId();
            }
        }

        return $result;
    }
}
