<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-06-14
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration;

use DateTime;
use Exception;
use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Data\CustomerInterface as SyncedCustomer;
use Fastmag\Sync\Api\Data\CustomerInterfaceFactory as SyncedCustomerFactory;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\CustomerExistsException;
use Fastmag\Sync\Exception\InactiveCustomerException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Process\Entity\ToMagento\Customer as CustomerEntity;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToMagento\Integration as IntegrationTrait;
use Magento\Customer\Api\AccountManagementInterface as AccountManager;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Customer\Api\Data\CustomerInterface as CustomerModel;
use Magento\Customer\Api\Data\CustomerInterfaceFactory as CustomerFactory;
use Magento\Customer\Api\Data\GroupInterface as CustomerGroupModel;
use Magento\Customer\Api\Data\GroupInterfaceFactory as CustomerGroupFactory;
use Magento\Customer\Api\GroupManagementInterface as CustomerGroupManager;
use Magento\Customer\Api\GroupRepositoryInterface as CustomerGroupRepository;
use Magento\Customer\Model\ResourceModel\GroupRepository as CustomerGroupResourceRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\DB\Adapter\Pdo\Mysql;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Newsletter\Model\SubscriptionManagerInterface as SubscriptionManager;
use Magento\Store\Api\GroupRepositoryInterface as StoreGroupRepository;
use Magento\Store\Api\StoreRepositoryInterface as StoreRepository;
use Magento\Store\Api\WebsiteRepositoryInterface as WebsiteRepository;
use Magento\Store\Model\ScopeInterface;

/**
 * Class Save
 *
 * Integration worker for tomagento_integration_customer_create and tomagento_integration_customer_update jobs
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Customer extends Worker
{
    use IntegrationTrait;

    /** @inheritDoc */
    protected $code = 'tomagento_integration_customer';

    /** @var SyncedCustomerFactory $syncedCustomerFactory */
    protected $syncedCustomerFactory;

    /** @var SyncedCustomerRepository $syncedCustomerRepository */
    protected $syncedCustomerRepository;

    /** @var CustomerRepository $customerRepository */
    protected $customerRepository;

    /** @var CustomerFactory $customerFactory */
    protected $customerFactory;

    /** @var WebsiteRepository $websiteRepository */
    protected $websiteRepository;

    /** @var CustomerGroupManager $customerGroupManager */
    protected $customerGroupManager;

    /** @var SubscriptionManager $subscriptionManager */
    protected $subscriptionManager;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var CustomerGroupRepository $customerGroupRepository */
    protected $customerGroupRepository;

    /** @var CustomerGroupFactory $customerGroupFactory */
    protected $customerGroupFactory;

    /** @var StoreGroupRepository $storeGroupRepository */
    protected $storeGroupRepository;

    /** @var StoreRepository $storeRepository */
    protected $storeRepository;

    /** @var AccountManager $accountManager */
    protected $accountManager;

    /** @var string $hydrationWorker */
    protected $hydrationWorker = 'tomagento_hydration_customer';

    /** @var string[] $subordinateWorkersAfter */
    protected $subordinateWorkersAfter = ['tomagento_integration_customer_address'];

    /** @var Job $currentJob */
    protected $currentJob;

    /**
     * Save constructor
     *
     * @param Logger                   $logger
     * @param JobRepository            $jobRepository
     * @param Config                   $config
     * @param SyncedCustomerFactory    $syncedCustomerFactory
     * @param SyncedCustomerRepository $syncedCustomerRepository
     * @param CustomerRepository       $customerRepository
     * @param CustomerFactory          $customerFactory
     * @param WebsiteRepository        $websiteRepository
     * @param CustomerGroupManager     $customerGroupManager
     * @param SubscriptionManager      $subscriptionManager
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param CustomerGroupRepository  $customerGroupRepository
     * @param CustomerGroupFactory     $customerGroupFactory
     * @param StoreGroupRepository     $storeGroupRepository
     * @param StoreRepository          $storeRepository
     * @param AccountManager           $accountManager
     *
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        SyncedCustomerFactory $syncedCustomerFactory,
        SyncedCustomerRepository $syncedCustomerRepository,
        CustomerRepository $customerRepository,
        CustomerFactory $customerFactory,
        WebsiteRepository $websiteRepository,
        CustomerGroupManager $customerGroupManager,
        SubscriptionManager $subscriptionManager,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CustomerGroupRepository $customerGroupRepository,
        CustomerGroupFactory $customerGroupFactory,
        StoreGroupRepository $storeGroupRepository,
        StoreRepository $storeRepository,
        AccountManager $accountManager
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->config = $config;
        $this->syncedCustomerFactory = $syncedCustomerFactory;
        $this->syncedCustomerRepository = $syncedCustomerRepository;
        $this->customerRepository = $customerRepository;
        $this->customerFactory = $customerFactory;
        $this->websiteRepository = $websiteRepository;
        $this->customerGroupManager = $customerGroupManager;
        $this->subscriptionManager = $subscriptionManager;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->customerGroupRepository = $customerGroupRepository;
        $this->customerGroupFactory = $customerGroupFactory;
        $this->storeGroupRepository = $storeGroupRepository;
        $this->storeRepository = $storeRepository;
        $this->accountManager = $accountManager;
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_IMPORT_ENABLE);
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());

            return;
        }

        foreach ($this->getJobs()->getItems() as $job) {
            if (!$job->isInError()) {
                $this->currentJob = $job;

                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                } catch (InactiveCustomerException|CustomerExistsException $exception) {
                    $this->skipJob($job, $exception);
                }

                $this->saveJob($job);
            }
        }
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws CustomerExistsException
     * @throws InactiveCustomerException
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var CustomerEntity $entity */
        $entity = $job->getEntity();

        if (!$entity->getEmailAddress()) {
            throw new JobException(__(
                'Customer %1 does not have a valid email: %2',
                $entity->getFastmagId(),
                $entity->getData('email_address')
            ));
        }

        if (!$entity->isActive() && !$this->config->isSetFlag(Config::XML_PATH_CUSTOMER_IMPORT_INACTIVE_CUSTOMERS)) {
            throw new InactiveCustomerException(__('Customer %1 is not active on Fastmag.', $entity->getFastmagId()));
        }

        $customer = $this->checkIfCustomerExists($entity);
        if (!$customer) {
            $customer = $this->customerFactory->create();
            $this->saveEntity($customer, $entity);
        } else {
            throw new CustomerExistsException(
                __('Customer #%1 already exists in Magento, we do not update it.', $entity->getFastmagId())
            );
        }
    }

    /**
     * Check if a customer in Magento DB has the ID or the email address given in parameter
     * and returns it if it exists
     *
     * @param CustomerEntity $entity
     *
     * @return CustomerModel|false
     */
    protected function checkIfCustomerExists($entity)
    {
        try {
            $fastmagCustomer = $this->syncedCustomerRepository->getByFastmagId($entity->getFastmagId());

            if ($fastmagCustomer->getId()) {
                $customer = $this->customerRepository->getById($fastmagCustomer->getMagentoCustomerId());
            } else {
                $customer = $this->customerRepository->get($entity->getEmailAddress());
            }
        } catch (LocalizedException $exception) {
            $customer = false;
        }

        return $customer;
    }

    /**
     * Save customer with entity attributes
     *
     * @param CustomerModel  $customer
     * @param CustomerEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveEntity($customer, $entity)
    {
        $this->setCustomerData($customer, $entity);

        try {
            if ($customer->getId() === null && $this->config->isSetFlag(
                Config::XML_PATH_CUSTOMER_IMPORT_SEND_WELCOME_EMAIL,
                ScopeInterface::SCOPE_STORE,
                $customer->getStoreId()
            )) {
                $customer = $this->accountManager->createAccount($customer);
            } else {
                $customer = $this->customerRepository->save($customer);
            }
        } catch (LocalizedException $exception) {
            throw new JobException(__(
                'Unable to sync the customer: %1',
                $exception->getMessage()
            ));
        }

        $entity->setMagentoId($customer->getId())
            ->setMagentoDefaultBilling($customer->getDefaultBilling())
            ->setMagentoDefaultShipping($customer->getDefaultShipping());

        $this->saveNewsletterPreferences($customer, $entity);

        $this->saveCustomerSync($customer, $entity);
    }

    /**
     * Set entity data into customer model
     *
     * @param CustomerModel  $customer
     * @param CustomerEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function setCustomerData($customer, $entity)
    {
        $defaultWebsite = $this->websiteRepository->getDefault();
        try {
            $defaultStoreId = $this->storeGroupRepository->get($defaultWebsite->getDefaultGroupId())
                ->getDefaultStoreId();
        } catch (NoSuchEntityException $exception) {
            throw new JobException(__('Unable to find default store for website #%1', $defaultWebsite));
        }

        if ($customer->getWebsiteId() === null) {
            $customer->setWebsiteId($defaultWebsite->getId());
        }
        if ($customer->getStoreId() === null) {
            $customer->setStoreId($defaultStoreId);
        }

        $customer->setEmail($entity->getEmailAddress())
            ->setGender($entity->getGender())
            ->setPrefix($entity->getPrefix())
            ->setFirstname($entity->getFirstname())
            ->setLastname($entity->getLastname())
            ->setDob($entity->getDob())
            ->setCreatedAt($entity->getCreationDate());

        $customer->setConfirmation(null);

        $this->setGroupCustomer($customer, $entity);
    }

    /**
     * Try to set customer group to customer
     *
     * @param CustomerModel  $customer
     * @param CustomerEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function setGroupCustomer($customer, $entity)
    {
        try {
            $groupId = $this->getCustomerGroupId($entity);
            if (!$groupId) {
                $groupId = $this->customerGroupManager->getDefaultGroup()->getId();
            }

            $customer->setGroupId($groupId);
        } catch (JobException $exception) {
            throw $exception;
        } catch (LocalizedException $exception) {
            throw new JobException(__('Unable to set group to the customer: %1', $exception->getMessage()));
        }
    }

    /**
     * Get customer group ID
     *
     * @param CustomerEntity $entity
     *
     * @return int
     *
     * @throws JobException
     */
    public function getCustomerGroupId($entity)
    {
        $code = $this->getCustomerGroupCode($entity);

        $this->searchCriteriaBuilder->addFilter(CustomerGroupModel::CODE, $code);
        $searchCriteria = $this->searchCriteriaBuilder->create();

        try {
            $customerGroupList = $this->customerGroupRepository->getList($searchCriteria);
        } catch (LocalizedException $exception) {
            throw new JobException(__(
                'Unable to search for the customer group "%1": %2',
                $code,
                $exception->getMessage()
            ));
        }

        if ($customerGroupList->getTotalCount() > 1) {
            throw new JobException(__(
                'It seems there is more than one customer group called "%1". Please review them ASAP.',
                $code
            ));
        }

        if ($customerGroupList->getTotalCount() === 0) {
            $customerGroup = $this->customerGroupFactory->create();
            $customerGroup->setCode($code)
                ->setTaxClassId(CustomerGroupResourceRepository::DEFAULT_TAX_CLASS_ID);

            try {
                $customerGroup = $this->customerGroupRepository->save($customerGroup);

                $this->log($this->currentJob, 'Customer group "' . $code . '" is created.');
            } catch (LocalizedException $exception) {
                throw new JobException(__(
                    'Unable to save the new customer group "%1": %2',
                    $code,
                    $exception->getMessage()
                ));
            }

            $result = $customerGroup->getId();
        } else {
            $result = $customerGroupList->getItems()[0]->getId();
        }

        return $result;
    }

    /**
     * Get customer group code
     *
     * @param CustomerEntity $entity
     *
     * @return string
     */
    public function getCustomerGroupCode($entity)
    {
        $config = $this->config->getValue(Config::XML_PATH_CUSTOMER_IMPORT_CUSTOMER_GROUPS_FIELDS);
        $customerGroup = 'General';

        if ($entity->getDiscountRate() !== null
            && in_array($config, [Constants::VALUE_DISCOUNT, Constants::VALUE_BOTH], true)
        ) {
            $customerGroup = Constants::CUSTOMER_DISCOUNT_GROUP_PREFIX
                . number_format($entity->getDiscountRate(), 2, '.', '');
        }

        if ($entity->getGrouping() !== null
            && in_array($config, [Constants::VALUE_GROUPING, Constants::VALUE_BOTH], true)
        ) {
            $customerGroup = Constants::CUSTOMER_GROUPING_GROUP_PREFIX . $entity->getGrouping();
        }

        return $customerGroup;
    }

    /**
     * Try to set customer group to customer
     *
     * @param CustomerModel  $customer
     * @param CustomerEntity $entity
     *
     * @return void
     */
    protected function saveNewsletterPreferences($customer, $entity)
    {
        if ($entity->getSubscribeNewsletter()) {
            $this->subscriptionManager->subscribeCustomer($customer->getId(), $customer->getStoreId());

            $this->log(
                $this->currentJob,
                'Customer #' . $customer->getId() . ' has subscribed to newsletters of store #'
                . $customer->getStoreId()
            );
        } else {
            foreach ($this->storeRepository->getList() as $store) {
                $this->subscriptionManager->unsubscribeCustomer($customer->getId(), $store->getId());
            }

            $this->log(
                $this->currentJob,
                'Customer #' . $customer->getId() . ' has unsubscribed to newsletters of all stores'
            );
        }
    }

    /**
     * Add a line in synced customers table
     *
     * @param CustomerModel  $customer
     * @param CustomerEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveCustomerSync($customer, $entity)
    {
        $syncedCustomer = $this->getSyncedCustomer($customer, $entity);
        $syncedCustomer->setMagentoCustomerId($customer->getId())
            ->setFastmagCustomerId($entity->getFastmagId());

        try {
            $now = new DateTime();
            $now = $now->format(Mysql::DATETIME_FORMAT);
        } catch (Exception $exception) {
            $now = date(Mysql::DATETIME_FORMAT);
        }
        $syncedCustomer->setProcessedAt($now);

        try {
            $this->syncedCustomerRepository->save($syncedCustomer);
        } catch (CouldNotSaveException $exception) {
            throw new JobException(__($exception->getMessage()));
        }
    }

    /**
     * Get corresponding line on synced customers table
     *
     * @param CustomerModel  $customer
     * @param CustomerEntity $entity
     *
     * @return SyncedCustomer
     *
     * @throws JobException
     */
    protected function getSyncedCustomer($customer, $entity)
    {
        try {
            $syncedCustomer = $this->syncedCustomerRepository->getByMagentoId($customer->getId());
        } catch (NoSuchEntityException $exception) {
            $syncedCustomer = false;
        } catch (LocalizedException $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        if (!$syncedCustomer) {
            try {
                $syncedCustomer = $this->syncedCustomerRepository->getByFastmagId($entity->getFastmagId());
            } catch (NoSuchEntityException $exception) {
                $syncedCustomer = false;
            } catch (LocalizedException $exception) {
                throw new JobException(__($exception->getMessage()));
            }

            if ($syncedCustomer && $syncedCustomer->getId()) {
                throw new JobException(__(
                    'Customer with Fastmag ID "%1" already synced with customer #%2, but just saved as customer #%3',
                    $entity->getFastmagId(),
                    $syncedCustomer->getId(),
                    $customer->getId()
                ));
            }
        }

        if (!$syncedCustomer) {
            $syncedCustomer = $this->syncedCustomerFactory->create();
        }

        return $syncedCustomer;
    }
}
