<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-01-25
 ******************************************************************************/

namespace Fastmag\Sync\Model;

use Exception;
use Fastmag\Sync\Api\CustomerRepositoryInterface;
use Fastmag\Sync\Api\Data\CustomerInterface;
use Fastmag\Sync\Api\Data\CustomerInterfaceFactory;
use Fastmag\Sync\Api\Data\CustomerSearchResultsInterfaceFactory;
use Fastmag\Sync\Model\ResourceModel\Customer as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Customer\CollectionFactory;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Reflection\DataObjectProcessor;

/**
 * Class CustomerRepository
 *
 * Repository of synchronized customers model
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class CustomerRepository implements CustomerRepositoryInterface
{
    /** @var ResourceModel $resource */
    protected $resource;

    /** @var CustomerFactory $customerFactory */
    protected $customerFactory;

    /** @var CollectionFactory $collectionFactory */
    protected $collectionFactory;

    /** @var CustomerSearchResultsInterfaceFactory $searchResultsFactory */
    protected $searchResultsFactory;

    /** @var DataObjectHelper $dataObjectHelper */
    protected $dataObjectHelper;

    /** @var DataObjectProcessor $dataObjectProcessor */
    protected $dataObjectProcessor;

    /** @var CustomerInterfaceFactory $customerInterfaceFactory */
    protected $customerInterfaceFactory;

    /** @var CollectionProcessorInterface $collectionProcessor */
    private $collectionProcessor;

    /**
     * CustomerRepository constructor
     *
     * @param ResourceModel                         $resource
     * @param CustomerFactory                       $customerFactory
     * @param CustomerInterfaceFactory              $customerInterfaceFactory
     * @param CollectionFactory                     $collectionFactory
     * @param CustomerSearchResultsInterfaceFactory $searchResultsFactory
     * @param DataObjectHelper                      $dataObjectHelper
     * @param DataObjectProcessor                   $dataObjectProcessor
     * @param CollectionProcessorInterface|null     $collectionProcessor
     */
    public function __construct(
        ResourceModel $resource,
        CustomerFactory $customerFactory,
        CustomerInterfaceFactory $customerInterfaceFactory,
        CollectionFactory $collectionFactory,
        CustomerSearchResultsInterfaceFactory $searchResultsFactory,
        DataObjectHelper $dataObjectHelper,
        DataObjectProcessor $dataObjectProcessor,
        CollectionProcessorInterface $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->customerFactory = $customerFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->dataObjectHelper = $dataObjectHelper;
        $this->customerInterfaceFactory = $customerInterfaceFactory;
        $this->dataObjectProcessor = $dataObjectProcessor;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(CustomerInterface $customer)
    {
        try {
            $this->resource->save($customer);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $customer;
    }

    /**
     * @inheritDoc
     */
    public function getById($customerId)
    {
        $customer = $this->customerFactory->create();
        $this->resource->load($customer, $customerId);
        if (!$customer->getId()) {
            throw new NoSuchEntityException(__('The customer with the "%1" ID doesn\'t exist.', $customerId));
        }

        return $customer;
    }

    /**
     * @inheritDoc
     */
    public function getByMagentoId($fastmagId)
    {
        $customer = $this->customerFactory->create();
        $this->resource->loadByMagentoId($customer, $fastmagId);
        if (!$customer->getId()) {
            throw new NoSuchEntityException(__('The customer with the Magento ID "%1" doesn\'t exist.', $fastmagId));
        }

        return $customer;
    }

    /**
     * @inheritDoc
     */
    public function getByFastmagId($fastmagId)
    {
        $customer = $this->customerFactory->create();
        $this->resource->loadByFastmagId($customer, $fastmagId);
        if (!$customer->getId()) {
            throw new NoSuchEntityException(__('The customer with the Fastmag ID "%1" doesn\'t exist.', $fastmagId));
        }

        return $customer;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $criteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($criteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(CustomerInterface $customer)
    {
        try {
            $this->resource->delete($customer);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($entityId)
    {
        return $this->delete($this->getById($entityId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessorInterface
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
