<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-10-25
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\RemoteSync;

use DateTime;
use DateTimeZone;
use Exception;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Process\Manager\RemoteSync as Manager;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\Clean as CleanTrait;
use Fastmag\Sync\Process\Worker\FastmagSql as FastmagSqlTrait;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\Pdo\Mysql;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Sync
 *
 * Abstract class inherited by all remote sync workers
 */
abstract class Sync extends Worker
{
    use CleanTrait;
    use FastmagSqlTrait;

    /** @var string */
    public const ENTITY_JOB_CODE_PREFIX = 'tomagento_integration_';

    /** @var int $queriesCount */
    protected $queriesCount = 1;

    /** @var int $jobsCount */
    protected $jobsCount = 0;

    /**
     * Sync constructor.
     *
     * @param Logger             $logger
     * @param ResourceConnection $resourceConnection
     * @param Proxy              $fastmagSql
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Proxy $fastmagSql
    ) {
        parent::__construct($logger);

        $this->resourceConnection = $resourceConnection;
        $this->fastmagSql = $fastmagSql;
    }

    /**
     * @inheritDoc
     *
     * @throws CouldNotSaveException
     */
    public function run()
    {
        $lastExecutionDate = $this->getLastExecutionDate();

        for ($i = 0; $i < $this->queriesCount; $i++) {
            $sql = $this->getSqlQuery($lastExecutionDate, $i);

            try {
                $result = $this->getFastmagSqlConnection()->get($sql);
            } catch (NoConnectionException $exception) {
                $this->logger->alert(
                    __('Unable to connect with Fastmag database. Error: %1', $exception->getMessage())->render()
                );
                $result = [];
            } catch (Exception $exception) {
                $this->logger->error(
                    __('Unable to perform the query. Message: %1- Query : %2', $exception->getMessage(), $sql)->render()
                );
                $result = [];
            }

            $newLastExecution = $this->getDate();

            foreach ($result as $event) {
                $newLastExecution = $this->saveJob($event);
            }

            $newLastExecution = $this->getDate($newLastExecution);

            if (isset($newLastExecution)) {
                $this->setLastExecutionDate($newLastExecution);
            }
        }
    }

    /**
     * Returns the SQL query to be
     *
     * @param string $lastExecutionDate
     * @param int    $bulkCnt
     *
     * @return string
     */
    protected function getSqlQuery($lastExecutionDate, $bulkCnt)
    {
        $limit = '';
        if ($this->queriesCount > 1) {
            $limit = ' LIMIT ' . ($bulkCnt * Manager::SQL_LIMIT) . ', ' . Manager::SQL_LIMIT;
        }

        return 'SELECT identifiant, job, code_mag, date_creation, date_maj, date_a_traiter, priorite, commentaire
            FROM fmsync_job_queue
            WHERE date_maj > \'' . $lastExecutionDate . '\' AND ' . $this->getSqlCondition() . '
            ORDER BY date_maj
            ' . $limit;
    }

    /**
     * Returns specific SQL condition to sync jobs
     *
     * @return string
     */
    abstract protected function getSqlCondition();

    /**
     * Save event as job in Magento DB and return the creation date as last execution date for the next sync
     *
     * @param array $event
     *
     * @return string
     *
     * @throws CouldNotSaveException
     */
    abstract protected function saveJob($event);

    /**
     * @inheritDoc
     *
     * To avoid annoying exception catching on DateTime constructor, only returns formatted date as string
     *
     * @return string
     */
    public function getLastExecutionDate()
    {
        $select = $this->resourceConnection->getConnection()
            ->select()
            ->from('fastmag_sync_job_last_execution', 'last_executed_at')
            ->where('job_code = \'' . $this->getCode() . '\'');

        $value = $this->resourceConnection->getConnection()->fetchOne($select);

        try {
            if ($value === false) {
                $result = new DateTime('1970-01-01', new DateTimeZone('UTC'));
            } else {
                $result = new DateTime($value, new DateTimeZone('UTC'));
                $result->setTimezone(new DateTimeZone('Europe/Paris'));
            }
        } catch (Exception $exception) {
            $result = $value;
        }

        if (get_class($result) === 'DateTime') {
            $result = $result->format(Mysql::DATETIME_FORMAT);
        }

        return $result;
    }

    /**
     * @inheritDoc
     *
     * To avoid annoying exception catching on DateTime constructor, only returns formatted date as string
     *
     * @param DateTime|string $date
     *
     * @return string
     */
    protected function getUtcDateTimeFromFastmag($date)
    {
        try {
            $dateTime = new DateTime($date, new DateTimeZone('Europe/Paris'));
            $dateTime->setTimezone(new DateTimeZone('UTC'));
        } catch (Exception $exception) {
            $dateTime = $date;
        }

        if (get_class($dateTime) === 'DateTime') {
            $dateTime = $dateTime->format(Mysql::DATETIME_FORMAT);
        }

        return $dateTime;
    }

    /**
     * Get jobs count in Fastmag
     *
     * @return int
     */
    public function getJobsCount()
    {
        $lastExecutionDate = $this->getLastExecutionDate();

        $sql = 'SELECT COUNT(*) AS jobs_count
        FROM fmsync_job_queue
        WHERE date_maj > \'' . $lastExecutionDate . '\'
            AND ' . $this->getSqlCondition();

        try {
            $result = $this->getFastmagSqlConnection()->get($sql);

            if (array_key_exists('jobs_count', $result[0])) {
                $this->jobsCount = (int)$result[0]['jobs_count'];
            }
        } catch (NoConnectionException $exception) {
            $this->logger->alert(
                __('Unable to connect with Fastmag database. Error: %1', $exception->getMessage())->render()
            );
            $this->jobsCount = 0;
        } catch (Exception $exception) {
            $this->logger->error(
                __('Unable to perform the query. Message: %1 - Query : %2', $exception->getMessage(), $sql)->render()
            );
            $this->jobsCount = 0;
        }

        return $this->jobsCount;
    }

    /**
     * Set queries count
     *
     * @param int $queriesCount
     *
     * @return $this
     */
    public function setQueriesCount($queriesCount)
    {
        $this->queriesCount = $queriesCount;

        return $this;
    }

    /**
     * Generate Magento-style job code given the Fastmag job code
     *
     * @see https://stackoverflow.com/questions/1993721/how-to-convert-pascalcase-to-pascal-case
     *
     * @param string $fastmagJobCode
     *
     * @return string
     */
    public function generateJobCode($fastmagJobCode)
    {
        preg_match_all('!([A-Z][A-Z0-9]*(?=$|[A-Z][a-z0-9])|[A-Za-z][a-z0-9]+)!', $fastmagJobCode, $matches);
        $ret = $matches[0];

        foreach ($ret as &$match) {
            $match = $match === strtoupper($match) ? strtolower($match) : lcfirst($match);
        }

        return self::ENTITY_JOB_CODE_PREFIX . implode('_', $ret);
    }

    /**
     * Get date
     *
     * @param string|null $date
     *
     * @return DateTime|string
     */
    protected function getDate($date = null)
    {
        try {
            $result = new DateTime($date);
        } catch (Exception $exception) {
            $result = date(Mysql::DATETIME_FORMAT);
        }

        return $result;
    }
}
