<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-10-04
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Order;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Data\OrderInterfaceFactory as SyncedOrderFactory;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\OrderRepositoryInterface as SyncedOrderRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Helper\Data as Helper;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\Item as OrderItemEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Transfer;
use Fastmag\Sync\Process\Worker\FastmagSql as FastmagSqlTrait;
use Magento\Framework\Serialize\Serializer\Json;
use stdClass;

/**
 * Class OrderTransfer
 *
 * Integration worker used for sending inventory transfer in Fastmag
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class OrderTransfer extends Transfer
{
    use FastmagSqlTrait;

    /** @inheritDoc */
    protected $code = 'tofastmag_integration_order_transfer';

    /** @var SyncedOrderFactory $syncedOrderFactory */
    protected $syncedOrderFactory;

    /** @var SyncedOrderRepository $syncedOrderRepository */
    protected $syncedOrderRepository;

    /** @var array $currentTransfers */
    protected $currentTransfers;

    /**
     * OrderTransfer constructor
     *
     * @param Logger                $logger
     * @param JobRepository         $jobRepository
     * @param Api                   $api
     * @param Config                $config
     * @param Json                  $jsonSerializer
     * @param FastmagSql            $fastmagSql
     * @param SyncedOrderFactory    $syncedOrderFactory
     * @param SyncedOrderRepository $syncedOrderRepository
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Api $api,
        Config $config,
        Json $jsonSerializer,
        FastmagSql $fastmagSql,
        SyncedOrderFactory $syncedOrderFactory,
        SyncedOrderRepository $syncedOrderRepository
    ) {
        parent::__construct($logger, $jobRepository, $api, $config, $jsonSerializer);

        $this->fastmagSql = $fastmagSql;
        $this->syncedOrderFactory = $syncedOrderFactory;
        $this->syncedOrderRepository = $syncedOrderRepository;
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        $this->currentJob = $job;

        $transactionType = $this->currentJob->getEntity()->getTransactionType();

        if ($this->transactionTypeNeedsTransfers($transactionType)) {
            $itemsStockLevels = $this->checkForTransfers();

            if (count($itemsStockLevels) > 0) {
                foreach ($itemsStockLevels as $itemId => $itemStockLevels) {
                    $this->calculateTransfers($itemId, $itemStockLevels);
                }

                foreach ($this->currentTransfers as $stockCode => $productsToTransfer) {
                    $this->sendTransfer($stockCode, $productsToTransfer);
                }
            }
        }
    }

    /**
     * Tells if a transaction type needs inventory transfers before
     *
     * @param string $transactionType
     *
     * @return bool
     */
    protected function transactionTypeNeedsTransfers($transactionType)
    {
        return in_array($transactionType, [
            OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
            OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION
        ], true);
    }

    /**
     * Check on Fastmag if the quantity ordered require stock transfers on Fastmag DB
     *
     * @return array
     *
     * @throws JobException
     */
    protected function checkForTransfers()
    {
        $result = [];

        $stockCodes = $this->currentJob->getEntity()->getAllFastmagStocks();

        $items = $this->currentJob->getEntity()->getItems();

        if (count($items) > 0) {
            /** @var OrderItemEntity $item */
            foreach ($items as $itemId => $item) {
                $qtyOrdered = $item->getQtyOrdered();
                $barcode = $item->getFastmagBarcode();
                $size = $item->getFastmagSize();
                $color = $item->getFastmagColor();

                $inventoryLevels = $this->getFastmagInventoryLevels($barcode, $size, $color, $stockCodes);

                if ($inventoryLevels[$stockCodes[0]] < $qtyOrdered) {
                    $this->log(
                        $this->currentJob,
                        'Need transfer for the item #' . $item->getMagentoId()
                        . ', current inventory level for stock "' . $stockCodes[0] . '" is '
                        . $inventoryLevels[$stockCodes[0]] . ' (' . $qtyOrdered . ' ordered)'
                    );

                    $result[$itemId] = $inventoryLevels;
                }
            }
        }

        return $result;
    }

    /**
     * Get Fastmag inventory levels for the item
     *
     * @param string   $barcode
     * @param string   $size
     * @param string   $color
     * @param string[] $stockCodes
     *
     * @return array
     *
     * @throws JobException
     */
    protected function getFastmagInventoryLevels($barcode, $size, $color, $stockCodes)
    {
        $result = [];
        foreach ($stockCodes as $stockCode) {
            $result[$stockCode] = 0;
        }

        try {
            $conditionSizeColor = $this->getConditionSizeColor($size, $color);

            $sql = 'SELECT CodeMag AS stock_code, SUM(Stock) AS total_inventory
                FROM stock
                WHERE CodeMag IN (' . $this->getFastmagSqlConnection()->escape($stockCodes) . ')
                    AND AR = 1
                    AND BarCode = ' . $this->getFastmagSqlConnection()->escape($barcode) . '
                    ' . $conditionSizeColor . '
                GROUP BY CodeMag
                HAVING SUM(Stock) > 0';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new JobException(__(
                'Error when getting inventory level of order\'s items. Message: %2. Item barcode: %2. Item size: %3. Item color: %4. Stock code: %5',
                $exception->getMessage(),
                $barcode,
                $size,
                $color,
                implode(',', $stockCodes)
            ));
        }

        if (count($rows) > 0) {
            foreach ($rows as $row) {
                $result[$row['stock_code']] = (int)$row['total_inventory'];
            }
        }

        return $result;
    }

    /**
     * Get SQL condition of size and color when getting inventory levels on Fastmag
     *
     * @param string $size
     * @param string $color
     *
     * @return string
     *
     * @throws NoConnectionException
     */
    protected function getConditionSizeColor($size, $color)
    {
        $result = '';

        if ($size !== null) {
            $result .= ' AND Taille = ' . $this->getFastmagSqlConnection()->escape($size);
        }
        if ($color !== null) {
            $result .= ' AND Couleur = ' . $this->getFastmagSqlConnection()->escape($color);
        }

        return $result;
    }

    /**
     * Send transfers to Fastmag
     *
     * @param int   $itemId
     * @param array $itemStockLevels
     *
     * @return void
     *
     * @throws JobException
     */
    protected function calculateTransfers($itemId, $itemStockLevels)
    {
        $itemEntity = $this->currentJob->getEntity()->getItemByMagentoId($itemId);

        if ($itemEntity === null || count($this->currentJob->getEntity()->getItems()) === 0) {
            throw new JobException(
                __('Item #%1 can not be transferred as it does not exist in the order in the first place', $itemId)
            );
        }

        $qtyToTransfer = $itemEntity->getQtyOrdered();
        $qtyTransferred = 0;
        $shop = $this->currentJob->getEntity()->getFastmagShop();

        foreach ($itemStockLevels as $stockCode => $inventoryLevel) {
            if ($stockCode !== $shop && $inventoryLevel >= $qtyToTransfer) {
                $this->currentTransfers[$stockCode][] = [
                    'item'            => $itemEntity,
                    'qty_to_transfer' => $qtyToTransfer
                ];

                $qtyToTransfer = 0;
                break;
            }
        }

        if ($qtyToTransfer > 0) {
            foreach ($itemStockLevels as $stockCode => $inventoryLevel) {
                if ($stockCode !== $shop && $inventoryLevel > 0) {
                    $qtyTransferred = min($qtyToTransfer, $inventoryLevel);

                    $this->currentTransfers[$stockCode][] = [
                        'item'            => $itemEntity,
                        'qty_to_transfer' => $qtyTransferred
                    ];

                    $qtyToTransfer -= $qtyTransferred;
                }

                if ($qtyToTransfer === 0) {
                    break;
                }
            }
        }
    }

    /**
     * Send transfer to Fastmag
     *
     * @param string $stockCode
     * @param array  $productsToTransfer
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendTransfer($stockCode, $productsToTransfer)
    {
        $endpoint = '/boa/transfert/create/index.ips';
        $request = $this->generateTransferDataRequest($stockCode, $productsToTransfer);

        try {
            $requestAt = date('Y-m-d H:i:s');

            $response = $this->api->post($endpoint, $request);
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send inventory transfer for order #%1 on Fastmag through API. Code: %2. Message: %3. Cause: %4. Data sent: %5',
                $this->currentJob->getEntity()->getMagentoId(),
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                $exception->getRequest()
            ));
        }

        if (array_key_exists('data', $response)) {
            $response = $response['data'][0];
        }

        if ($response['status'] === 'OK') {
            if ($this->config->isSetFlag(Config::XML_PATH_ORDER_ALERT_ENABLE_FOR_TRANSFER)) {
                $this->sendAlert(
                    $this->currentJob->getEntity()->getFastmagShop(),
                    $stockCode,
                    $productsToTransfer
                );
            }

            $this->saveSyncedTransfer(
                $request,
                $response,
                $requestAt,
                $endpoint,
                $stockCode,
                $this->currentJob->getEntity()->getFastmagShop()
            );
        } else {
            throw new JobException(__(
                'Unable to send inventory transfer for order #%1 on Fastmag through API. Message: %2. Data sent: %3',
                $this->currentJob->getEntity()->getMagentoId(),
                $response['Message'],
                $this->jsonSerializer->serialize($request)
            ));
        }
    }

    /**
     * Generate API request for transfer
     *
     * @param string $stockCode
     * @param array  $productsToTransfer
     *
     * @return stdClass
     */
    protected function generateTransferDataRequest($stockCode, $productsToTransfer)
    {
        $totalQtyToTransfer = 0;
        $productData = [];

        foreach ($productsToTransfer as $productToTransfer) {
            $itemEntity = $productToTransfer['item'];
            $qtyToTransfer = (int)$productToTransfer['qty_to_transfer'];

            $productData[] = (object)[
                'Reference'           => '#' . $itemEntity->getFastmagId(),
                'Designation'         => Helper::removeAccents($itemEntity->getName()),
                'PrixUnitaire'        => Helper::formatPrice($itemEntity->getPriceInclTax()),
                'Quantite'            => $qtyToTransfer,
                'TotalLigneTransfert' => Helper::formatPrice($itemEntity->getPriceInclTax() * $qtyToTransfer),
            ];

            $totalQtyToTransfer += $qtyToTransfer;
        }

        $result = [
            'MagasinOrigine'     => $stockCode,
            'MagasinDestination' => $this->currentJob->getEntity()->getFastmagShop(),
            'TotalQuantite'      => $totalQtyToTransfer,
            'Vendeur'            => $this->currentJob->getEntity()->getFastmagSeller(),
            'MAJStock'           =>
                $this->config->isSetFlag(Config::XML_PATH_ORDER_WORKFLOW_INVENTORY_TRANSFER) ? 'OUI' : 'NON',
            'Produits'           => $productData
        ];

        return (object)['Transferts' => [(object)$result]];
    }

    /**
     * Send alert to shop
     *
     * @param string $senderShop
     * @param string $recipientShop
     * @param array  $itemsData
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendAlert($senderShop, $recipientShop, $itemsData)
    {
        $alertMessage = [];

        $sendProductName = $this->config->isSetFlag(Config::XML_PATH_ORDER_ALERT_DISPLAY_NAME_TRANSFER);

        foreach ($itemsData as $itemData) {
            $message = '';
            $itemEntity = $itemData['item'];
            $qtyTransferred = (int)$itemData['qty_to_transfer'];

            if ($sendProductName) {
                $message = 'Désignation : ' . $itemEntity->getName() . ' - ';
            }

            $message .= 'Ref : ' . $itemEntity->getFastmagBarcode() . ' - '
                . 'Couleur : ' . $itemEntity->getFastmagColor() . ' - '
                . 'Taille : ' . $itemEntity->getFastmagSize() . ' - '
                . 'Quantité : ' . $qtyTransferred;

            $alertMessage[] = $message;
        }

        $request = (object)[
            'Consignes' => [(object)[
                'Destinataire' => $recipientShop,
                'Sujet'        => __('Web order #%1', $this->currentJob->getEntity()->getMagentoId()),
                'Message'      => __('To prepare:{CR}{LF}%1', implode('{CR}{LF}', $alertMessage)),
                'Expediteur'   => $senderShop,
                'Slide'        => '1'
            ]]
        ];

        try {
            $response = $this->api->post('/boa/consigne/create/index.ips', $request);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException(__($message), $this->api->getLastRequest(), null, $response['errorCode']);
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send alert to shop %1. Code: %2. Message: %3. Cause: %4. Alert sent: %5',
                $this->currentJob->getEntity()->getFastmagShop(),
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                implode('{CR}{LF}', $alertMessage)
            ));
        }
    }
}
