<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-23
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\RemoteSync;

use DateInterval;
use DateTime;
use Exception;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\Clean as CleanTrait;
use Fastmag\Sync\Process\Worker\FastmagSql as FastmagSqlTrait;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\Pdo\Mysql;

/**
 * Class Clean
 *
 * Worker class used for cleaning obsolete lines from queue table in Fastmag DB
 */
class Clean extends Worker
{
    use CleanTrait;
    use FastmagSqlTrait;

    /** @inheritDoc */
    protected $code = 'remotesync_clean';

    /**
     * Clean constructor.
     *
     * @param Logger             $logger
     * @param ResourceConnection $resourceConnection
     * @param Proxy              $fastmagSql
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Proxy $fastmagSql
    ) {
        parent::__construct($logger);

        $this->resourceConnection = $resourceConnection;
        $this->fastmagSql = $fastmagSql;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $now = new DateTime();
            $lastCleaningDate = $this->getLastExecutionDate();

            if ($lastCleaningDate->format(Mysql::DATE_FORMAT) === '1970-01-01') {
                $lastCleaningDate = new DateTime();
                $lastCleaningDate->sub(new DateInterval('PT14H'));
            } else {
                $lastCleaningDate->add(new DateInterval('PT12H'));
            }
        } catch (Exception $exception) {
            return;
        }

        if ($now > $lastCleaningDate) {
            $this->logger->debug(
                'Clean old events twice a day - ' . $lastCleaningDate->format(Mysql::DATETIME_FORMAT)
            );

            $query = 'DELETE FROM fmsync_job_queue
                WHERE date_maj < \'' . $lastCleaningDate->format(Mysql::DATE_FORMAT) . '\'';

            $this->logger->debug($query);

            try {
                $result = $this->getFastmagSqlConnection()->post($query);

                if ($result) {
                    $this->setLastExecutionDate();
                }
            } catch (NoConnectionException $exception) {
                $this->logger->alert(
                    __('Unable to connect with Fastmag database. %1', $exception->getMessage())->render()
                );
            }
        }
    }
}
