<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-12-02
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;

use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Exception\ProductNotSyncableException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;

/**
 * Class Data
 *
 * Default hydration worker for product
 */
class Data extends Product
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product';

    /** @var array $colorsI18n */
    protected $colorsI18n;

    /** @var array $sizesI18n */
    protected $sizesI18n;

    /**
     * @inheritDoc
     */
    public function run()
    {
        $this->getSyncConfig();
        $this->getGlobalData();

        foreach ($this->getJobs()->getItems() as $job) {
            $this->currentJob = $job;

            try {
                $entity = $this->getDataFromFastmag();

                $this->hydrateJob($job, $entity);
            } catch (JobException $exception) {
                $this->invalidateJob($job, $exception);
            } catch (ProductNotSyncableException $exception) {
                $this->skipJob($job, $exception);
            }

            $this->saveJob($job);
        }
    }

    /**
     * @inheritDoc
     */
    protected function initiateCurrentEntity($fastmagRef)
    {
        $this->currentEntity = $this->productEntityFactory->create();
        $this->currentEntity->setRef($fastmagRef)
            ->setColorsI18n($this->colorsI18n)
            ->setSizesI18n($this->sizesI18n)
            ->setMagentoId($this->getMagentoProductId($fastmagRef));
    }

    /**
     * @inheritDoc
     */
    protected function getDataByStore()
    {
        foreach (array_keys($this->syncConfig['stores']) as $storeId) {
            $productI18nEntity = $this->productEntityFactory->create();
            $this->currentEntity->addI18n($productI18nEntity, $storeId);

            $this->getFastmagShopTaxRate($storeId);
            $this->getI18nData($storeId);
            $this->getChildrenData($storeId);
        }
    }

    /**
     * Get global data (independent of the jobs)
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getGlobalData()
    {
        $this->getColorsI18n();
        $this->getSizesI18n();
    }

    /**
     * Get colors labels i18n
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getColorsI18n()
    {
        try {
            $sql = 'SELECT Langue AS language, Couleur AS color, Traduction AS translation FROM couleurslangues';

            $this->colorsI18n = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new ProcessException(__($exception->getMessage()));
        }
    }

    /**
     * Get sizes labels i18n
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getSizesI18n()
    {
        try {
            $sql = 'SELECT Langue AS language, Taille AS size, Traduction AS translation FROM tailleslangues';

            $this->sizesI18n = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new ProcessException(__($exception->getMessage()));
        }
    }

    /**
     * @inheritDoc
     */
    protected function getParentData()
    {
        $additionalAttributes = $this->getAdditionalAttributes('pf');

        try {
            $sql = 'SELECT pf.VisibleWeb AS visible_web,
                    pf.Actif AS active,
                    COUNT(s.ID) AS stock_level,
                    GROUP_CONCAT(DISTINCT IF(Stock >= 1, s.CodeMag, NULL)) AS stocks_list,
                    f.Famille AS vat_family,
                    f.TauxTVA AS vat_rate,
                    TRIM(pf.Marque) AS brand,
                    TRIM(pf.Fournisseur) AS supplier,
                    IF(pf.Poids > 0, pf.Poids, 0.1) AS weight,
                    TRIM(pf.Designation) AS designation,
                    TRIM(pf.Designation2) AS designation_bis
                    ' .  $additionalAttributes . '
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode = '
                . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                    AND s.AR = 1
                GROUP BY pf.BarCode';

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if (count($rows) === 0) {
                $sql = 'SELECT pf.VisibleWeb AS visible_web,
                    pf.Actif AS active,
                    0 AS stock_level,
                    \'\' AS stocks_list,
                    f.Famille AS vat_family,
                    f.TauxTVA AS vat_rate,
                    TRIM(pf.Marque) AS brand,
                    TRIM(pf.Fournisseur) AS supplier,
                    IF(pf.Poids > 0, pf.Poids, 0.1) AS weight,
                    TRIM(pf.Designation) AS designation,
                    TRIM(pf.Designation2) AS designation_bis
                    ' .  $additionalAttributes . '
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode = '
                    . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                GROUP BY pf.BarCode';

                $rows = $this->getFastmagSqlConnection()->get($sql);
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        $this->log($this->currentJob, 'SQL parent query: ' . $sql);

        if (count($rows) > 0) {
            $row = reset($rows);

            $this->currentEntity->addData($row);
        }
    }

    /**
     * Get tax rate for the product
     *
     * @param int   $storeId
     * @param float $defaultValue
     *
     * @return void
     *
     * @throws JobException
     */
    protected function getFastmagShopTaxRate($storeId, $defaultValue = 20.0)
    {
        $result = $defaultValue;

        $fastmagShop = $this->getSyncConfigValueFromArray(Config::XML_PATH_ORDER_WORKFLOW_SHOP, $storeId);

        try {
            $sql = 'SELECT TVA AS shop_vat_rate FROM mag WHERE Code = '
                . $this->getFastmagSqlConnection()->escape($fastmagShop);

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        if (count($rows) > 0) {
            $row = reset($rows);

            $result = (float)$row['shop_vat_rate'];
        }

        $currentProductI18nEntity = $this->currentEntity->getI18n($storeId);

        if ($currentProductI18nEntity !== null) {
            $currentProductI18nEntity->setShopVatRate($result);
        }
    }

    /**
     * @inheritDoc
     */
    protected function getChildrenDataSql($storeId = null)
    {
        $fastmagShop = $this->getSyncConfigValueFromArray(Config::XML_PATH_ORDER_WORKFLOW_SHOP, $storeId);
        $rateCode = $this->getSyncConfigValueFromArray(Config::XML_PATH_PRICE_IMPORT_FASTMAG_RATE, $storeId);

        $standardPriceSubquery = $this->getStandardPriceSubquery($fastmagShop, $rateCode);
        $combinationPriceSubquery = $this->getCombinationPriceSubquery($rateCode);
        $ratePriceSubquery = $this->getRatePriceSubquery($rateCode);
        $salePriceSubquery = $this->getSalePriceSubquery($fastmagShop);
        $variationSalePriceSubquery = $this->getVariationSalePriceSubquery($fastmagShop);
        $buyingPriceSubquery = $this->getBuyingPriceSubquery();

        $stockCondition = $this->getStocksCodesCondition($storeId);

        return 'SELECT
                (
                    SELECT CONCAT(
                        \'{"\', \'fastmag_id\', \'": \', Produit,
                        \', "\', \'ean13\', \'": "\', RIGHT(GenCod, 13),
                        \'", "\', \'supplier_ref\', \'": "\', RefFournisseur,
                        \'", "\', \'weight\', \'": \', IF(PoidsTC > 0, PoidsTC, 0.1), \'}\'
                    )
                    FROM produits AS p
                    WHERE p.BarCode = stock.BarCode AND p.Taille = stock.Taille AND p.Couleur = stock.Couleur
                    LIMIT 1
                ) AS combination_data,
                (
                    SELECT ProduitActif
                    FROM produits AS p2
                    WHERE p2.BarCode = stock.BarCode AND p2.Taille = stock.Taille AND p2.Couleur = stock.Couleur
                    LIMIT 1
                ) AS active,
                (
                    SELECT 1
                    FROM stock AS s2
                    WHERE s2.BarCode = stock.BarCode AND s2.Taille = stock.Taille AND s2.Couleur = stock.Couleur
                        ' . $stockCondition . '
                    LIMIT 1
                ) AS active_stock,
                (
                    SELECT IFNULL(ValeurInteger, -1) AS no_web
                    FROM produits AS p3 LEFT JOIN complementsr AS c
                        ON p3.Produit = c.Clef AND c.Champ = \'NO_WEB\' AND c.Nature = \'PRODUITSTC\'
                    WHERE p3.BarCode = stock.BarCode AND p3.Taille = stock.Taille AND p3.Couleur = stock.Couleur
                    LIMIT 1
                ) AS no_web,
                pf.PrixVente AS indicative_price,
                (' . $standardPriceSubquery . ') AS standard_price,
                (' . $combinationPriceSubquery . ') AS combination_price,
                (' . $ratePriceSubquery . ') AS rate_price,
                (' . $salePriceSubquery . ') AS sale_price,
                (' . $variationSalePriceSubquery . ') AS variation_sale_price,
                (' . $buyingPriceSubquery . ') AS buying_price,
                TRIM(stock.Taille) AS size,
                TRIM(stock.Couleur) AS color,
                (
                    SELECT TRIM(Descriptif)
                    FROM produitscouleurs AS pc
                    WHERE pc.Barcode = stock.BarCode AND pc.Couleur = stock.Couleur
                ) AS color_description,
                SUM(Stock) AS stock_level,
                stock.AR AS stock_received
            FROM produitsfiches AS pf INNER JOIN stock ON pf.BarCode = stock.BarCode
            WHERE stock.AR = 1
                AND stock.BarCode =
                ' . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                ' . $stockCondition . '
            GROUP BY stock.Barcode, stock.Taille, stock.Couleur
            HAVING active = 1 AND (no_web < 1 OR no_web IS NULL) ' . $this->getOnlyDefinedStocksCondition($storeId) . '
            ORDER BY stock.Couleur, stock.Taille';
    }

    /**
     * @inheritDoc
     */
    protected function getOnlyDefinedStocksCondition($storeId = null)
    {
        $result = '';

        if ($this->getSyncConfigValueFromArray(Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS, $storeId)) {
            $result .= ' AND active_stock = 1 ';
        }

        return $result;
    }

    /**
     * Get all i18nized data from Fastmag
     *
     * @param int $storeId
     *
     * @return void
     *
     * @throws JobException
     */
    protected function getI18nData($storeId)
    {
        $defaultLanguage = $this->getSyncConfigValueFromArray(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_LANGUAGE);
        $language = $this->getSyncConfigValueFromArray(
            Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_LANGUAGE,
            $storeId,
            false
        );

        if ($language !== $defaultLanguage) {

            try {
                $sql = $this->getI18nDataSql($language);
                $rows = $this->getFastmagSqlConnection()->get($sql);
            } catch (Exception $exception) {
                throw new JobException(__($exception->getMessage()));
            }

            if (count($rows) > 0) {
                $row = reset($rows);

                $currentProductI18nEntity = $this->currentEntity->getI18n($storeId);
                if ($currentProductI18nEntity === null) {
                    throw new JobException(__('No data available for store #%1', $storeId));
                }

                foreach ($row as $code => $value) {
                    if (!empty($value) && $value !== $this->currentEntity->getData($code)) {
                        $currentProductI18nEntity->addData($row);
                    }
                }
            }
        }
    }

    /**
     * Get SQL query getting the i18ns data
     *
     * @param string $language
     *
     * @return string
     *
     * @throws NoConnectionException
     */
    protected function getI18nDataSql($language)
    {
        $additionalAttributes = $this->getAdditionalAttributes('pl', false);

        return 'SELECT Designation AS designation,
                Designation2 AS designation2
                ' . $additionalAttributes . '
            FROM produitslangues AS pl
            WHERE BarCode = ' . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                AND TRIM(CONCAT_WS(Designation, Designation2, Modele, Matiere, Rayon, Famille, SsFamille, Saison, Descriptif, Theme)) <> \'\'
                AND Langue = ' . $this->getFastmagSqlConnection()->escape($language, true);
    }

    /**
     * Returns SQL to get data for additionnal attributes, given config fields
     *
     * @param string $alias
     * @param bool   $forParent
     *
     * @return string
     */
    protected function getAdditionalAttributes($alias, $forParent = true)
    {
        $additionalAttributes = [];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_DESCRIPTION)) {
            $additionalAttributes[] = $alias . '.Descriptif AS description';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_FAMILY)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.Famille) AS family';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SUBFAMILY)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.SsFamille) AS subfamily';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SECTION)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.Rayon) AS section';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SEASON)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.Saison) AS season';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MODEL)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.Modele) AS model';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MATERIAL)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.Matiere) AS material';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_THEME)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.Theme) AS theme';
        }

        if ($forParent) {
            $additionalAttributes = array_merge(
                $additionalAttributes,
                $this->getParentAdditionalAttributesAsArray($alias)
            );
        }

        return (count($additionalAttributes) > 0 ? ', ' . implode(', ', $additionalAttributes) : '');
    }

    /**
     * Returns as array the list of parent only additional attributes to sync, if the config allows it
     *
     * @param string $alias
     *
     * @return string[]
     */
    protected function getParentAdditionalAttributesAsArray($alias)
    {
        $result = [];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_CREATION_DATE)) {
            $result[] = $alias . '.DateCreation AS creation_date';
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_TEXTILE)) {
            $result[] = 'TRIM(' . $alias . '.Lavage) AS washing';
            $result[] = 'TRIM(' . $alias . '.Chlore) AS chlorine';
            $result[] = 'TRIM(' . $alias . '.Repassage) AS ironing';
            $result[] = 'TRIM(' . $alias . '.Pressing) AS dry_cleaning';
            $result[] = 'TRIM(' . $alias . '.Sechage) AS drying';
            $result[] = 'TRIM(' . $alias . '.AquaNettoyage) AS water';
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_IS_NEW)) {
            $result[] = $alias . '.EnNouveaute AS is_new';
        }

        return $result;
    }
}
