<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-01-25
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoScheduledInterface;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Jobqueue\ToMagentoFactory;
use Fastmag\Sync\Model\Jobqueue\ToMagentoRepository;
use Fastmag\Sync\Model\Jobqueue\ToMagentoScheduledRepository;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\Clean as CleanTrait;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Api\WebsiteRepositoryInterface as WebsiteRepository;
use Magento\Store\Model\ScopeInterface;

/**
 * Class Scheduled
 *
 * Worker class used for moving scheduled jobs to the current job queue
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Scheduled extends Worker
{
    use CleanTrait;

    /** @inheritDoc */
    protected $code = 'tomagento_move_scheduled';

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var ToMagentoFactory $jobFactory */
    protected $jobFactory;

    /** @var ToMagentoRepository $jobRepository */
    protected $jobRepository;

    /** @var ToMagentoScheduledRepository $jobScheduledRepository */
    protected $jobScheduledRepository;

    /** @var Config $config */
    protected $config;

    /** @var WebsiteRepository $websiteRepository */
    protected $websiteRepository;

    /**
     * Scheduled constructor.
     *
     * @param Logger                       $logger
     * @param ResourceConnection           $resourceConnection
     * @param SearchCriteriaBuilder        $searchCriteriaBuilder
     * @param ToMagentoFactory             $jobFactory
     * @param ToMagentoRepository          $jobRepository
     * @param ToMagentoScheduledRepository $jobScheduledRepository
     * @param Config                       $config
     * @param WebsiteRepository            $websiteRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        ToMagentoFactory $jobFactory,
        ToMagentoRepository $jobRepository,
        ToMagentoScheduledRepository $jobScheduledRepository,
        Config $config,
        WebsiteRepository $websiteRepository
    ) {
        parent::__construct($logger);

        $this->resourceConnection = $resourceConnection;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->jobFactory = $jobFactory;
        $this->jobRepository = $jobRepository;
        $this->jobScheduledRepository = $jobScheduledRepository;
        $this->config = $config;
        $this->websiteRepository = $websiteRepository;
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(ToMagentoScheduledInterface::SCHEDULED_AT, date('Y-m-d H:i:s'), 'lt')
            ->create();

        $jobsScheduled = $this->jobScheduledRepository->getList($searchCriteria);

        foreach ($jobsScheduled->getItems() as $jobScheduled) {
            try {
                $job = $this->jobRepository->getByUniqueKey(
                    $jobScheduled->getContentId(),
                    $jobScheduled->getJobCode(),
                    10
                );

                $job->setProcessedAt(null)
                    ->setContent($job->getContent());
            } catch (NoSuchEntityException $exception) {
                $job = $this->jobFactory->create([
                    ToMagentoInterface::CONTENT_ID => $jobScheduled->getContentId(),
                    ToMagentoInterface::JOB_CODE   => $jobScheduled->getJobCode(),
                    ToMagentoInterface::CREATED_AT => $jobScheduled->getCreatedAt(),
                    ToMagentoInterface::CONTENT    => $jobScheduled->getContent(),
                    ToMagentoInterface::PRIORITY   => 10
                ]);
            }

            try {
                $this->jobRepository->save($job);

                $this->jobScheduledRepository->delete($jobScheduled);
            } catch (Exception $exception) {
                throw new ProcessException(__($exception->getMessage()));
            }
        }

        $this->deleteByShops();

        $this->setLastExecutionDate();
    }

    /**
     * Delete scheduled jobs for shops not defined in config
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function deleteByShops()
    {
        $shopsList = $this->getShopsList();

        if (count($shopsList) > 0) {
            $searchCriteria = $this->searchCriteriaBuilder
                ->addFilter(ToMagentoScheduledInterface::FASTMAG_SHOP, $shopsList, 'nin')
                ->create();

            $jobsToDelete = $this->jobScheduledRepository->getList($searchCriteria);

            foreach ($jobsToDelete->getItems() as $job) {
                try {
                    $this->jobScheduledRepository->delete($job);
                } catch (CouldNotDeleteException $exception) {
                    throw new ProcessException(__($exception->getMessage()));
                }
            }
        }
    }

    /**
     * Get the shops list from the config
     *
     * @return array
     */
    protected function getShopsList()
    {
        $result = [];

        $result[] = $this->config->getValue(Config::XML_PATH_ORDER_WORKFLOW_SHOP);

        foreach ($this->websiteRepository->getList() as $website) {
            $result[] = $this->config->getValue(
                Config::XML_PATH_ORDER_WORKFLOW_SHOP,
                ScopeInterface::SCOPE_WEBSITES,
                $website->getId()
            );
        }

        return array_unique($result);
    }
}
