<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-06-07
 ******************************************************************************/

namespace Fastmag\Sync\Cron;

use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Sql\DirectSql;
use Laminas\Uri\Uri as UriHandler;
use Magento\Framework\App\ProductMetadataInterface as ProductMetadata;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\HTTP\ZendClient;
use Magento\Framework\Module\ResourceInterface as ModuleResource;
use Magento\Framework\Webapi\Rest\Request;
use Magento\Store\Model\StoreManagerInterface;
use Zend_Http_Client_Exception;

/**
 * Class PushVersion
 *
 * Cron class used for log rotation
 */
class PushVersion
{
    /** @var Config $config */
    protected $config;

    /** @var ZendClient $client */
    protected $client;

    /** @var ModuleResource */
    protected $moduleResource;

    /** @var StoreManagerInterface */
    protected $storeManager;

    /** @var DirectSql $connection */
    protected $connection;

    /** @var UriHandler $uriHandler */
    protected $uriHandler;

    /** @var ProductMetadata $productMetadata */
    protected $productMetadata;

    /**
     * PushVersion constructor.
     *
     * @param Config                $config
     * @param ZendClient            $client
     * @param ModuleResource        $moduleResource
     * @param StoreManagerInterface $storeManager
     * @param DirectSql             $connection
     * @param UriHandler            $uriHandler
     * @param ProductMetadata       $productMetadata
     */
    public function __construct(
        Config $config,
        ZendClient $client,
        ModuleResource $moduleResource,
        StoreManagerInterface $storeManager,
        DirectSql $connection,
        UriHandler $uriHandler,
        ProductMetadata $productMetadata
    ) {
        $this->config = $config;
        $this->client = $client;
        $this->moduleResource = $moduleResource;
        $this->storeManager = $storeManager;
        $this->connection = $connection;
        $this->uriHandler = $uriHandler;
        $this->productMetadata = $productMetadata;
    }

    /**
     * Send module version to monitoring website
     *
     * @return void
     *
     * @throws NoSuchEntityException|NoConnectionException
     */
    public function execute()
    {
        $websiteHmioId = $this->config->getValue(Config::XML_PATH_ABOUT_WEBSITE_ID);
        $customer = 'fastmag_' . strtolower($this->config->getValue(Config::XML_PATH_CONNECT_API_CHAIN));
        $url = $this->storeManager->getStore()->getBaseUrl();
        $urlParsed = $this->uriHandler->parse($url);

        try {
            $this->client->setUri('https://www.home-made.io/monitoring/fastmagsync_modules_list.php')
                ->setMethod(Request::METHOD_POST)
                ->setHeaders(['Content-Type' => 'application/x-www-form-urlencoded'])
                ->setParameterPost([
                    'version'     => $this->moduleResource->getDbVersion('Fastmag_Sync'),
                    'cms'         => 'Magento 2',
                    'cms_version' => $this->productMetadata->getVersion(),
                    'website'     => $urlParsed->getScheme() . '://' .  $urlParsed->getHost(),
                    'id_site'     => $websiteHmioId,
                    'path'        => $urlParsed->getPath(),
                    'customer'    => $customer,
                    'mysql_use_fm_triggers' => (int)$this->connection->nativeTriggersExists()
                ]);

            $this->client->request();
        } catch (Zend_Http_Client_Exception $exception) {
            throw new NoConnectionException(__('Unable to push module version: %1', $exception->getMessage()));
        }

        $response = $this->client->getLastResponse();

        if ($response && ($response->getStatus() < 200 || $response->getStatus() >= 300)) {
            throw new NoConnectionException(__(
                'Unable to push module version. Status: %1, message: %2',
                $response->getStatus(),
                $response->getMessage()
            ));
        }
    }
}
