<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-12-02
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag;

use DateInterval;
use DateTime;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Jobqueue as AbstractJob;
use Fastmag\Sync\Model\Jobqueue\ToFastmagRepository as JobRepository;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\Clean as CleanTrait;
use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\Search\FilterGroupBuilder;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Clean
 *
 * Worker class used for cleaning to Fastmag jobs finished since more than 45 minutes
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Clean extends Worker
{
    use CleanTrait;

    /** @inheritDoc */
    protected $code = 'tofastmag_clean';

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /** @var FilterGroupBuilder $filterGroupBuilder */
    protected $filterGroupBuilder;

    /** @var FilterBuilder $filterBuilder */
    protected $filterBuilder;

    /**
     * Clean constructor.
     *
     * @param Logger                $logger
     * @param ResourceConnection    $resourceConnection
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param JobRepository         $jobRepository
     * @param FilterGroupBuilder    $filterGroupBuilder
     * @param FilterBuilder         $filterBuilder
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        JobRepository $jobRepository,
        FilterGroupBuilder $filterGroupBuilder,
        FilterBuilder $filterBuilder
    ) {
        parent::__construct($logger);

        $this->resourceConnection = $resourceConnection;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->jobRepository = $jobRepository;
        $this->filterGroupBuilder = $filterGroupBuilder;
        $this->filterBuilder = $filterBuilder;
    }

    /**
     * @inheritDoc
     *
     * @throws CouldNotSaveException
     */
    public function run()
    {
        $this->resetErrorJobs();
        $this->cleanDoneJobs();
    }

    /**
     * Update jobs data to allow rerun
     *
     * @throws CouldNotSaveException
     */
    protected function resetErrorJobs()
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(Job::RETRY_COUNT, 4, 'lt')
            ->addFilter(Job::STATUS, AbstractJob::STATUS_ERROR)
            ->create();

        $jobsToReset = $this->jobRepository->getList($searchCriteria);

        foreach ($jobsToReset->getItems() as $job) {
            $this->jobRepository->reset($job);
            $this->jobRepository->save($job);
        }
    }

    /**
     * Clean done jobs from queue
     *
     * @throws ProcessException
     */
    protected function cleanDoneJobs()
    {
        $limitDatetime = new DateTime();
        $limitDatetime->sub(new DateInterval('PT45M'));

        $statusSuccessFilter = $this->filterBuilder->setField(Job::STATUS)
            ->setValue(AbstractJob::STATUS_SUCCESS)
            ->setConditionType('eq')
            ->create();
        $statusSkippedFilter = $this->filterBuilder->setField(Job::STATUS)
            ->setValue(AbstractJob::STATUS_SKIPPED)
            ->setConditionType('eq')
            ->create();

        $statusFilter = $this->filterGroupBuilder
            ->addFilter($statusSuccessFilter)
            ->addFilter($statusSkippedFilter)
            ->create();

        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(Job::PROCESSED_AT, $limitDatetime->format('Y-m-d H:i:s'), 'lt')
            ->setFilterGroups([$statusFilter])
            ->create();

        $jobsToDelete = $this->jobRepository->getList($searchCriteria);

        foreach ($jobsToDelete->getItems() as $job) {
            try {
                $this->jobRepository->delete($job);
            } catch (CouldNotDeleteException $exception) {
                throw new ProcessException(__($exception->getMessage()));
            }
        }

        $this->setLastExecutionDate();
    }
}
