<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-09-19
 ******************************************************************************/

namespace Fastmag\Sync\Process\Entity\ToMagento\Product;

use Fastmag\Sync\Process\Entity\Date as DateTrait;
use Fastmag\Sync\Process\EntityInterface;
use Magento\Framework\DataObject;

/**
 * Class Variation
 *
 * Entity class for products child.
 * Save and transform data from Fastmag for Magento easy integration,
 * and transfer it from hydration worker to integration worker
 *
 * @method int|null getFastmagId()
 * @method string|null getSupplierRef()
 * @method string|null getSize()
 * @method string|null getColor()
 * @method string|null getColorDescription()
 * @method float|null getWeight()
 * @method string|null getSaleReason()
 * @method string|null getVariationSaleReason()
 */
class Variation extends DataObject implements EntityInterface
{
    use DateTrait;

    /**
     * Product constructor.
     *
     * @param array $data
     */
    public function __construct(array $data = [])
    {
        parent::__construct($data);

        $this->timeZoneInput = 'Europe/Paris';
        $this->timeZoneOutput = 'Europe/Paris';

        $this->configureDateFormatter();
    }

    /**
     * Get EAN13
     *
     * @return string|null
     */
    public function getEan13()
    {
        return $this->getData('ean13');
    }

    /**
     * Get active
     *
     * @return bool|null
     */
    public function getActive()
    {
        return $this->getData('active') !== null ? (bool)$this->getData('active') : null;
    }

    /**
     * Get active stock
     *
     * @return bool|null
     */
    public function getActiveStock()
    {
        return $this->getData('active_stock') !== null ? (bool)$this->getData('active_stock') : null;
    }

    /**
     * Get no web
     *
     * @return bool|null
     */
    public function getNoWeb()
    {
        return $this->getData('no_web') !== null ? (int)$this->getData('no_web') : null;
    }

    /**
     * Get indicative price
     *
     * @return float|null
     */
    public function getIndicativePrice()
    {
        return $this->getData('indicative_price') !== null ? (float)$this->getData('indicative_price') : null;
    }

    /**
     * Get standard price
     *
     * @return float|null
     */
    public function getStandardPrice()
    {
        return $this->getData('standard_price') !== null ? (float)$this->getData('standard_price') : null;
    }

    /**
     * Get combination price
     *
     * @return float|null
     */
    public function getCombinationPrice()
    {
        return $this->getData('combination_price') !== null ? (float)$this->getData('combination_price') : null;
    }

    /**
     * Get rate price
     *
     * @return float|null
     */
    public function getRatePrice()
    {
        return $this->getData('rate_price') !== null ? (float)$this->getData('rate_price') : null;
    }

    /**
     * Get sale price
     *
     * @return float|null
     */
    public function getSalePrice()
    {
        $result = $this->getData('sale_price');

        return ($result !== null && $result !== -1.0) ? (float)$result : null;
    }

    /**
     * Get sale discount
     *
     * @return float|null
     */
    public function getSaleDiscount()
    {
        $result = $this->getData('sale_discount');

        return ($result !== null && $result !== 0.0) ? (float)$result : null;
    }

    /**
     * Get sale beginning date
     *
     * @return string|null
     */
    public function getSaleBeginAt()
    {
        return $this->getFormattedDate('sale_begin_at', 'y/MM/dd HH:mm:ss');
    }

    /**
     * Get sale end date
     *
     * @return string|null
     */
    public function getSaleEndAt()
    {
        return $this->getFormattedDate('sale_end_at', 'y/MM/dd HH:mm:ss');
    }

    /**
     * Get variation sale price
     *
     * @return float|null
     */
    public function getVariationSalePrice()
    {
        $result = $this->getData('variation_sale_price');

        return ($result !== null && $result !== -1.0) ? (float)$result : null;
    }

    /**
     * Get variation sale discount
     *
     * @return float|null
     */
    public function getVariationSaleDiscount()
    {
        $result = $this->getData('variation_sale_discount');

        return ($result !== null && $result !== 0.0) ? (float)$result : null;
    }

    /**
     * Get variation sale beginning date
     *
     * @return string|null
     */
    public function getVariationSaleBeginAt()
    {
        return $this->getFormattedDate('variation_sale_begin_at', 'y/MM/dd HH:mm:ss');
    }

    /**
     * Get variation sale end date
     *
     * @return string|null
     */
    public function getVariationSaleEndAt()
    {
        return $this->getFormattedDate('variation_sale_end_at', 'y/MM/dd HH:mm:ss');
    }

    /**
     * Get buying price
     *
     * @return float|null
     */
    public function getBuyingPrice()
    {
        return $this->getData('buying_price') !== null ? (float)$this->getData('buying_price') : null;
    }

    /**
     * Get stock level
     *
     * @return int|null
     */
    public function getStockLevel()
    {
        return $this->getData('stock_level') !== null ? (int)$this->getData('stock_level') : null;
    }

    /**
     * Is stock received in Fastmag
     *
     * @return bool|null
     */
    public function isStockReceived()
    {
        return $this->getData('stock_received') !== null ? (bool)$this->getData('stock_received') : null;
    }

    /**
     * Get i18ns
     *
     * @return Variation[]
     */
    public function getI18ns()
    {
        return $this->getData('i18ns') !== [] ?  $this->getData('i18ns') : null;
    }

    /**
     * Get i18n
     *
     * @param int $storeId
     *
     * @return Variation
     */
    public function getI18n($storeId)
    {
        $i18ns = $this->getData('i18ns');

        return $i18ns[$storeId] ?? null;
    }

    /**
     * Returns i18n of a specific code for a specific store
     *
     * @param string $code
     * @param int    $storeId
     *
     * @return mixed
     */
    public function getDataI18n($code, $storeId)
    {
        $result = null;

        $i18n = $this->getI18n($storeId);

        if ($i18n !== null) {
            $result = $i18n->getData($code);
        }

        return $result;
    }

    /**
     * Returns all i18ns for a specific code
     *
     * @param string $code
     *
     * @return array
     */
    public function getDataI18ns($code)
    {
        $result = [];

        $i18ns = $this->getData('i18ns');

        if (is_array($i18ns) && count($i18ns) > 0) {
            /** @var Variation $i18n */
            foreach ($i18ns as $storeId => $i18n) {
                if ($i18n->getData($code) !== null) {
                    $result[$storeId] = $i18n->getData($code);
                }
            }
        }

        return $result;
    }

    /**
     * Add stock level
     *
     * @param int $stockLevel
     *
     * @return Variation
     */
    public function addStockLevel($stockLevel)
    {
        return $this->setData('stock_level', (int)$this->getData('stock_level') + $stockLevel);
    }

    /**
     * Add i18n
     *
     * @param Variation $i18n
     * @param string    $storeId
     *
     * @return Variation
     */
    public function addI18n($i18n, $storeId)
    {
        $i18ns = $this->getData('i18ns');

        if ($i18ns === null) {
            $i18ns = [];
        }

        $i18ns[(int)$storeId] = $i18n;

        return $this->setData('i18ns', $i18ns);
    }

    /**
     * Check if a specific data is different from the standard value for a specific store.
     * If the data does not exist for a specific store, we consider that it's the same value as the standard one
     *
     * @param string $code
     * @param int    $storeId
     *
     * @return bool
     */
    public function isI18nDifferent($code, $storeId)
    {
        $result = false;

        $i18n = $this->getI18n($storeId);

        if ($i18n !== null) {
            $storeData = $i18n->getDataI18n($code, $storeId);

            if ($storeData !== null && $storeData !== $this->getData($code)) {
                $result = true;
            }
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    public function export()
    {
        $result = [
            'fastmag_id'              => $this->getFastmagId(),
            'ean13'                   => $this->getEan13(),
            'supplier_ref'            => $this->getSupplierRef(),
            'active'                  => $this->getActive(),
            'active_stock'            => $this->getActiveStock(),
            'is_no_web'               => $this->getNoWeb(),
            'size'                    => $this->getSize(),
            'color'                   => $this->getColor(),
            'color_description'       => $this->getColorDescription(),
            'weight'                  => $this->getWeight(),
            'indicative_price'        => $this->getIndicativePrice(),
            'standard_price'          => $this->getStandardPrice(),
            'combination_price'       => $this->getCombinationPrice(),
            'rate_price'              => $this->getRatePrice(),
            'sale_price'              => $this->getSalePrice(),
            'sale_discount'           => $this->getSaleDiscount(),
            'sale_begin_at'           => $this->getSaleBeginAt(),
            'sale_end_at'             => $this->getSaleEndAt(),
            'sale_reason'             => $this->getSaleReason(),
            'variation_sale_price'    => $this->getVariationSalePrice(),
            'variation_sale_discount' => $this->getVariationSaleDiscount(),
            'variation_sale_begin_at' => $this->getVariationSaleBeginAt(),
            'variation_sale_end_at'   => $this->getVariationSaleEndAt(),
            'variation_sale_reason'   => $this->getVariationSaleReason(),
            'buying_price'            => $this->getBuyingPrice(),
            'stock_level'             => $this->getStockLevel(),
            'is_stock_received'       => $this->isStockReceived(),
            'i18ns'                   => [],
        ];

        $i18ns = $this->getData('i18ns');
        if (is_array($i18ns) && count($i18ns) > 0) {
            foreach ($i18ns as $storeId => $i18n) {
                $result['i18ns'][$storeId] = $i18n->export();
            }
        }

        return array_filter(
            $result,
            static function ($value) {
                return $value !== null && $value !== [];
            }
        );
    }
}
