<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2023 HOMEMADE.IO SAS
 * @date      2023-03-01
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer\Address;

use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\AddressFactory as AddressEntityFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\CustomerFactory as CustomerEntityFactory;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration as HydrationTrait;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Archiving
 *
 * Hydration class used for archiving customer addresses in Fastmag when deleted in Magento
 */
class Archiving extends Worker
{
    use HydrationTrait;

    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_customer_address_archiving';

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /** @var Job $currentJob */
    protected $currentJob;

    /** @var CustomerEntityFactory $customerEntityFactory */
    protected $customerEntityFactory;

    /** @var AddressEntityFactory $addressEntityFactory */
    protected $addressEntityFactory;

    /** @var SyncedCustomerRepository $syncedCustomerRepository */
    protected $syncedCustomerRepository;

    /**
     * Archiving constructor
     *
     * @param Logger                   $logger
     * @param JobRepository            $jobRepository
     * @param Config                   $config
     * @param Json                     $jsonSerializer
     * @param AddressEntityFactory     $addressEntityFactory
     * @param CustomerEntityFactory    $customerEntityFactory
     * @param SyncedCustomerRepository $syncedCustomerRepository
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        AddressEntityFactory $addressEntityFactory,
        CustomerEntityFactory $customerEntityFactory,
        SyncedCustomerRepository $syncedCustomerRepository
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->config = $config;
        $this->jsonSerializer = $jsonSerializer;
        $this->addressEntityFactory = $addressEntityFactory;
        $this->customerEntityFactory = $customerEntityFactory;
        $this->syncedCustomerRepository = $syncedCustomerRepository;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        foreach ($this->getJobs()->getItems() as $job) {
            $this->currentJob = $job;
            $customerEntity = $this->getDataFromMagento();

            if ($customerEntity->getFastmagId() === null || $customerEntity->getShippingAddress() === null) {
                $exception = new JobException(__('Unable to format hydrated data as an correct entity'));
                $this->invalidateJob($job, $exception);
            } else {
                $job->setEntity($customerEntity);
            }

            $this->saveJob($job);
        }
    }

    /**
     * @inheritDoc
     */
    protected function getDataFromMagento()
    {
        // TODO: change (un)serializer logic for jobs to always get unserialized data and save serialized data
        $jobHydratedData = $this->jsonSerializer->unserialize($this->currentJob->getHydratedData());

        $customerEntity = $this->customerEntityFactory->create(['data' => $jobHydratedData]);
        $customerEntity->setFastmagId($this->getCustomerFastmagId($customerEntity->getMagentoId()));

        return $customerEntity;
    }

    /**
     * Get customer Fastmag ID from synced customers table
     *
     * @param int $customerMagentoId
     *
     * @return int|null
     */
    protected function getCustomerFastmagId($customerMagentoId)
    {
        try {
            $syncedCustomer = $this->syncedCustomerRepository->getByMagentoId($customerMagentoId);

            $result = $syncedCustomer->getFastmagCustomerId();
        } catch (LocalizedException $exception) {
            $result = null;
        }

        return $result;
    }
}
