<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2023 HOMEMADE.IO SAS
 * @date      2023-02-23
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Order;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Extractor\Order\Items\ExtractorFactory as ItemsExtractorFactory;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\Item as ItemEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\ItemFactory as ItemEntityFactory;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\FastmagSql as FastmagSqlTrait;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration as HydrationTrait;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Sales\Api\Data\OrderInterface as OrderModel;
use Magento\Sales\Api\Data\OrderItemInterface as ItemModel;
use Magento\Sales\Api\OrderRepositoryInterface as OrderRepository;

/**
 * Class Items
 *
 * Hydration class used for inserting or updating orders items from Magento to Fastmag
 */
class Items extends Worker
{
    use HydrationTrait;
    use FastmagSqlTrait;

    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_order_items';

    protected OrderRepository $orderRepository;
    protected ItemsExtractorFactory $itemsExtractorFactory;
    protected ItemEntityFactory $itemEntityFactory;

    /** @var Job $currentJob */
    protected $currentJob;

    /**
     * Items constructor
     *
     * @param Logger                $logger
     * @param JobRepository         $jobRepository
     * @param Config                $config
     * @param Json                  $jsonSerializer
     * @param FastmagSql            $fastmagSql
     * @param OrderRepository       $orderRepository
     * @param ItemsExtractorFactory $itemsExtractorFactory
     * @param ItemEntityFactory     $itemEntityFactory
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        FastmagSql $fastmagSql,
        OrderRepository $orderRepository,
        ItemsExtractorFactory $itemsExtractorFactory,
        ItemEntityFactory $itemEntityFactory
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->config = $config;
        $this->jsonSerializer = $jsonSerializer;
        $this->fastmagSql = $fastmagSql;
        $this->orderRepository = $orderRepository;
        $this->itemsExtractorFactory = $itemsExtractorFactory;
        $this->itemEntityFactory = $itemEntityFactory;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        foreach ($this->getJobs()->getItems() as $job) {
            $this->currentJob = $job;

            try {
                $items = $this->getDataFromMagento();

                if (count($items) > 0) {
                    /** @var OrderEntity $orderEntity */
                    $orderEntity = $this->currentJob->getEntity();

                    foreach ($items as $item) {
                        $orderEntity->addItem($item);
                    }

                    $this->hydrateJob($job, $orderEntity);
                }
            } catch (JobException $exception) {
                $this->invalidateJob($job, $exception);
            }

            $this->saveJob($job);
        }
    }

    /**
     * @inheritDoc
     *
     * @return ItemEntity[]
     */
    protected function getDataFromMagento()
    {
        /** @var OrderEntity $orderEntity */
        $orderEntity = $this->currentJob->getEntity();
        $result = [];

        if (!in_array(
            $orderEntity->getTransactionType(),
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_NONE
            ],
            true
        )) {
            $orderModel = $this->getOrder($orderEntity->getMagentoId());

            $itemsToAdd = $this->getOrderItems($orderModel);

            foreach ($itemsToAdd as $item) {
                $itemEntity = $this->generateItemEntity($item);
                $this->addItem($itemEntity, $result);
            }

            $this->addFastmagProductDataToItems($result);
        }

        return $result;
    }

    /**
     * Get order object from Magento
     *
     * @param int $orderId
     *
     * @return OrderModel
     *
     * @throws JobException
     */
    protected function getOrder($orderId): OrderModel
    {
        try {
            return $this->orderRepository->get($orderId);
        } catch (Exception $exception) {
            throw new JobException(
                __('Unable to get the matching order %1. Cause: %2', $orderId, $exception->getMessage())
            );
        }
    }

    /**
     * Get the items to sync in Fastmag
     *
     * @param OrderModel $order
     *
     * @return ItemModel[]
     */
    protected function getOrderItems(OrderModel $order): array
    {
        $itemsExtractor = $this->itemsExtractorFactory->create(
            ['order' => $order]
        );

        return $itemsExtractor->extract();
    }

    /**
     * Generate item entity with order item data
     *
     * @param ItemModel $item
     *
     * @return ItemEntity
     */
    protected function generateItemEntity(ItemModel $item): ItemEntity
    {
        $result = $this->itemEntityFactory->create();

        $result->setMagentoId($item->getItemId())
            ->setParentItemId($item->getParentItemId())
            ->setFastmagId($item->getFastmagProductId())
            ->setName($item->getName())
            ->setPriceInclTax($item->getBasePriceInclTax())
            ->setQtyOrdered($item->getQtyOrdered())
            ->setQtyCanceled($item->getQtyCanceled())
            ->setDiscountAmount($item->getBaseDiscountAmount())
            ->setRowTotalInclTax($item->getBaseRowTotalInclTax() - $item->getBaseDiscountAmount());

        if ($item->getParentItemId()) {
            $parentItem = $item->getParentItem();

            if ($parentItem !== null && $parentItem->getProductType() === Configurable::TYPE_CODE) {
                $result->setPriceInclTax($parentItem->getBasePriceInclTax())
                    ->setDiscountAmount($parentItem->getBaseDiscountAmount())
                    ->setRowTotalInclTax($parentItem->getBaseRowTotalInclTax() - $parentItem->getBaseDiscountAmount());
            }
        }

        if ($result->getDiscountAmount() > 0) {
            $result->setComment(__('DISCOUNT: ') . $result->getDiscountAmount());
        }

        return $result;
    }

    /**
     * Add item to the list
     *
     * @param ItemEntity   $item
     * @param ItemEntity[] $result
     *
     * @return void
     */
    protected function addItem(ItemEntity $item, &$result)
    {
        if (array_key_exists($item->getFastmagId(), $result)) {
            $existingItem = $result[$item->getFastmagId()];

            $existingItem->setQtyOrdered($existingItem->getQtyOrdered() + $item->getQtyOrdered())
                ->setQtyCanceled($existingItem->getQtyCanceled() + $item->getQtyCanceled())
                ->setDiscountAmount($existingItem->getDiscountAmount() + $item->getDiscountAmount())
                ->setRowTotalInclTax($existingItem->getRowTotalInclTax() + $item->getRowTotalInclTax());
        } else {
            $result[$item->getFastmagId()] = $item;
        }
    }

    /**
     * Get basic data from Fastmag
     *
     * @param ItemEntity[] $items
     *
     * @return void
     *
     * @throws JobException
     */
    protected function addFastmagProductDataToItems(array $items)
    {
        $itemsFastmagIds = array_keys($items);

        try {
            $sql = 'SELECT Produit AS product_id, BarCode AS barcode, Taille AS size, Couleur AS color
                FROM produits
                WHERE Produit IN (' . $this->getFastmagSqlConnection()->escape($itemsFastmagIds) . ')';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (NoConnectionException $exception) {
            throw new JobException(__(
                'Error when trying to check order\'s items product ids in Fastmag. Message: %1. Product IDs: %2',
                $exception->getMessage(),
                array_keys($itemsFastmagIds)
            ));
        }

        if (count($rows) < count($items)) {
            throw new JobException(__('Unable to find corresponding products in Fastmag for some items of the order'));
        }

        foreach ($rows as $row) {
            $itemEntity = $items[$row['product_id']];

            if ($itemEntity !== null) {
                $itemEntity->setFastmagBarcode($row['barcode'])
                    ->setFastmagSize($row['size'])
                    ->setFastmagColor($row['color']);
            }
        }
    }
}
