<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2023 HOMEMADE.IO SAS
 * @date      2023-03-01
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration;

use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToFastmag\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\FastmagSql as FastmagSqlTrait;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration as IntegrationTrait;
use Magento\Customer\Api\AddressRepositoryInterface as AddressRepository;
use Magento\Framework\Exception\LocalizedException;
use Monolog\Logger as Monolog;
use stdClass;

/**
 * Class Address
 *
 * Abstract class for Address related Integration workers
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
abstract class Address extends Worker
{
    use IntegrationTrait;
    use FastmagSqlTrait;

    /** @var Config $config */
    protected $config;

    /** @var AddressRepository $addressRepository */
    protected $addressRepository;

    /** @var Job $currentJob */
    protected $currentJob;

    /**
     * Address constructor
     *
     * @param Logger            $logger
     * @param JobRepository     $jobRepository
     * @param Api               $api
     * @param FastmagSql        $fastmagSql
     * @param Config            $config
     * @param AddressRepository $addressRepository
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Api $api,
        FastmagSql $fastmagSql,
        Config $config,
        AddressRepository $addressRepository
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->api = $api;
        $this->fastmagSql = $fastmagSql;
        $this->config = $config;
        $this->addressRepository = $addressRepository;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());

            return;
        }

        foreach ($this->getJobs()->getItems() as $job) {
            if (!$job->isInError()) {
                $this->currentJob = $job;

                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                }

                $this->saveJob($job);
            }
        }
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        $entity = $this->getCustomerEntity($job);

        $shippingAddress = $entity->getShippingAddress();

        if ($shippingAddress) {
            $this->saveFastmagAddress($entity->getMagentoId(), $entity->getEmailAddress(), $shippingAddress);
        }
    }

    /**
     * Get customer entity from job
     *
     * @param Job $job
     *
     * @return CustomerEntity
     *
     * @throws JobException
     */
    abstract protected function getCustomerEntity($job);

    /**
     * Save address on Fastmag
     *
     * @param int           $magentoId
     * @param string        $emailAddress
     * @param AddressEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveFastmagAddress($magentoId, $emailAddress, $entity)
    {
        $addressData = $this->setDataRequest($emailAddress, $entity);

        try {
            $response = $this->api->post('/boa/client/deliveryaddress/index.ips', $addressData);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('message', $response) ? $response['message'] : '');

                throw new ApiException(
                    __($message),
                    $this->api->getLastRequest(),
                    null,
                    (array_key_exists('errorCode', $response) ? $response['errorCode'] : 0)
                );
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send address\' data of the customer #%1 on Fastmag through API. Code: %2. Message: %3. Cause: %4. Data sent: %5',
                $magentoId,
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                $exception->getRequest()
            ));
        }

        $this->saveFastmagAlias($entity->getMagentoId(), $entity->getAlias());
    }

    /**
     * Set the address entity as intented for the API
     *
     * @param string        $emailAddress
     * @param AddressEntity $entity
     *
     * @return stdClass
     */
    protected function setDataRequest($emailAddress, $entity)
    {
        $result = [
            'Email'            => $emailAddress,
            'AdresseLivraison' => $entity->getAlias(),
            'Nom'              => $entity->getFirstname() . ' ' . $entity->getLastname(),
            'Adresse1'         => $entity->getStreetOne(),
            'Adresse2'         => $entity->getStreetTwo(),
            'CodePostal'       => $entity->getPostcode(),
            'Ville'            => $entity->getCity(),
            'Pays'             => $entity->getCountryId(),
            'Societe'          => $entity->getCompany(),
            'Telephone'        => $entity->getPhoneNumber(),
            'Archiver'         => (string)$entity->getArchive()
        ];

        foreach ($result as $field => $data) {
            if ($data === null || $data === '') {
                unset($result[$field]);
            }
        }

        return (object)['Adresses' => [(object)$result]];
    }

    /**
     * Save address Fasmtag alias in M2 DB
     *
     * @param int    $magentoId
     * @param string $fastmagAlias
     *
     * @return void
     */
    protected function saveFastmagAlias($magentoId, $fastmagAlias)
    {
        if ($magentoId) {
            try {
                $address = $this->addressRepository->getById($magentoId);
            } catch (LocalizedException $exception) {
                return;
            }

            $address->setCustomAttribute(Constants::ATTRIBUTE_ADDRESS_ALIAS_CODE, $fastmagAlias);

            try {
                $this->addressRepository->save($address);
            } catch (LocalizedException $exception) {
                $this->log(
                    $this->currentJob,
                    '[ERROR] Unable to save address Fastmag alias "' . $fastmagAlias
                    . '" in DB. Plase add it ASAP, or it will be duplicated both in Fastmag and M2 '
                    . 'when this address will be used again',
                    Monolog::ALERT
                );
            }
        }
    }
}
