<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2023 HOMEMADE.IO SAS
 * @date      2023-05-10
 ******************************************************************************/

namespace Fastmag\Sync\Helper;

use Exception;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Magento\Backend\App\Area\FrontNameResolver;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Translate\Inline\StateInterface;
use Magento\Framework\Escaper;
use Magento\Framework\Mail\Template\TransportBuilder;
use Magento\Store\Model\Store;

/**
 * Class Email
 *
 * Helper class for preconfig email
 */
class Email extends AbstractHelper
{
    /** @var StateInterface $inlineTranslation */
    protected StateInterface $inlineTranslation;

    /** @var Escaper $escaper */
    protected Escaper $escaper;

    /** @var TransportBuilder $transportBuilder */
    protected TransportBuilder $transportBuilder;

    /** @var Logger $logger */
    protected Logger $logger;

    /** @var Config $config */
    protected Config $config;

    /** @var string $templateId */
    protected $templateId;

    /** @var string[] $templateVariables */
    protected $templateVariables;

    /**
     * Email constructor
     *
     * @param Context          $context
     * @param StateInterface   $inlineTranslation
     * @param Escaper          $escaper
     * @param TransportBuilder $transportBuilder
     * @param Logger           $logger
     * @param Config           $config
     */
    public function __construct(
        Context          $context,
        StateInterface   $inlineTranslation,
        Escaper          $escaper,
        TransportBuilder $transportBuilder,
        Logger           $logger,
        Config           $config
    ) {
        parent::__construct($context);

        $this->inlineTranslation = $inlineTranslation;
        $this->escaper = $escaper;
        $this->transportBuilder = $transportBuilder;
        $this->logger = $logger;
        $this->config = $config;
    }

    /**
     * Set template ID
     *
     * @param string $templateId
     *
     * @return Email
     */
    public function setTemplateId(string $templateId): Email
    {
        $this->templateId = $templateId;

        return $this;
    }

    /**
     * Set template variables
     *
     * @param array $templateVariables
     *
     * @return Email
     */
    public function setTemplateVariables(array $templateVariables): Email
    {
        $this->templateVariables = $templateVariables;

        return $this;
    }

    /**
     * Send email
     *
     * @return void
     */
    public function sendEmail(): void
    {
        try {
            $this->inlineTranslation->suspend();

            $sender = [
                'name'  => $this->escaper->escapeHtml($this->config->getValue('trans_email/ident_general/name')),
                'email' => $this->escaper->escapeHtml($this->config->getValue('trans_email/ident_general/email')),
            ];

            $transport = $this->transportBuilder->setTemplateIdentifier($this->templateId)
                ->setTemplateVars($this->templateVariables)
                ->setTemplateOptions([
                    'area'  => FrontNameResolver::AREA_CODE,
                    'store' => Store::DEFAULT_STORE_ID
                ])
                ->setFromByScope($sender)
                ->addTo('monitoring@home-made.io')
                ->getTransport();

            $transport->sendMessage();
            $this->inlineTranslation->resume();
        } catch (Exception $exception) {
            $this->logger->debug($exception->getMessage());
        }
    }
}
