<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2023 HOMEMADE.IO SAS
 * @date      2023-06-20
 ******************************************************************************/

namespace Fastmag\Sync\Model\System\Connection;

use Fastmag\Sync\Api\LogApiRepositoryInterface as LogApiRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\LogApiFactory;
use Laminas\Http\Exception\RuntimeException;
use Laminas\Http\Request;
use Laminas\Http\Response;
use Laminas\Uri\Exception\InvalidUriPartException;
use Laminas\Uri\Http as HttpUri;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\HTTP\LaminasClient as HttpClient;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Api
 *
 * Class encapsulating the Fastmag API-related method
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Api
{
    /** @var string */
    protected const API_APPSOURCE = 'Web';

    /** @var string */
    protected const API_VERSION = '1';

    /** @var Config $config */
    protected Config $config;

    /** @var HttpClient $client */
    protected HttpClient $client;

    /** @var HttpUri $uri */
    protected HttpUri $uri;

    /** @var Json $jsonSerializer */
    protected Json $jsonSerializer;

    /** @var LogApiRepository $logApiRepository */
    protected LogApiRepository $logApiRepository;

    /** @var LogApiFactory $logApiFactory */
    protected LogApiFactory $logApiFactory;

    /** @var Logger $logger */
    protected Logger $logger;

    /** @var string $host */
    protected $host;

    /** @var string $chain */
    protected $chain;

    /** @var string $shop */
    protected $shop;

    /** @var string $user */
    protected $user;

    /** @var string $password */
    protected $password;

    /** @var string $token */
    protected $token;

    /**
     * Api constructor
     *
     * @param Config           $config
     * @param HttpClient       $client
     * @param HttpUri          $uri
     * @param Json             $jsonSerializer
     * @param LogApiRepository $logApiRepository
     * @param LogApiFactory    $logApiFactory
     * @param Logger           $logger
     */
    public function __construct(
        Config           $config,
        HttpClient       $client,
        HttpUri          $uri,
        Json             $jsonSerializer,
        LogApiRepository $logApiRepository,
        LogApiFactory    $logApiFactory,
        Logger           $logger
    ) {
        $this->config = $config;
        $this->client = $client;
        $this->uri = $uri;
        $this->jsonSerializer = $jsonSerializer;
        $this->logApiRepository = $logApiRepository;
        $this->logApiFactory = $logApiFactory;
        $this->logger = $logger;
    }

    /**
     * Set API access credentials
     *
     * @param string $host
     * @param string $chain
     * @param string $shop
     * @param string $user
     * @param string $password
     *
     * @return Api
     */
    public function setCredentials($host, $chain, $shop, $user, $password): Api
    {
        $this->host = $host;
        $this->chain = $chain;
        $this->shop = $shop;
        $this->user = $user;
        $this->password = $password;

        return $this;
    }

    /**
     * Set credentials from config
     *
     * @return Api
     */
    public function setCredentialsFromConfig(): Api
    {
        $host = $this->config->getValue(Config::XML_PATH_CONNECT_API_HOST);
        $chain = $this->config->getValue(Config::XML_PATH_CONNECT_API_CHAIN);
        $shop = $this->config->getValue(Config::XML_PATH_CONNECT_API_SHOP);
        $user = $this->config->getValue(Config::XML_PATH_CONNECT_API_USER);
        $password = $this->config->getValue(Config::XML_PATH_CONNECT_API_PASSWORD);

        $this->setCredentials($host, $chain, $shop, $user, $password);

        return $this;
    }

    /**
     * Check if the credentials are well filled
     *
     * @return bool
     */
    public function areCredentialsFilled(): bool
    {
        return $this->host !== null
            && $this->chain !== null
            && $this->shop !== null
            && $this->user !== null
            && $this->password !== null;
    }

    /**
     * Returns authentication token
     *
     * @throws ApiException
     *
     * @return bool
     */
    public function authenticate(): bool
    {
        if (!$this->areCredentialsFilled()) {
            $this->setCredentialsFromConfig();
        }

        $this->client->setUri($this->getUriObject('/boa/auth/token/index.ips'))
            ->setMethod(Request::METHOD_POST)
            ->setHeaders(['Content-Type' => 'application/x-www-form-urlencoded'])
            ->setParameterPost([
                'db'            => $this->chain,
                'store'         => $this->shop,
                'Appsource'     => self::API_APPSOURCE,
                'user'          => $this->user,
                'pass'          => $this->password,
                'FastmagApiVer' => self::API_VERSION,
            ]);

        try {
            $response = $this->client->send();
        } catch (RuntimeException $exception) {
            throw new ApiException(__('Unable to send the request to the API'), '', $exception);
        }

        $responseData = $this->getResponseData($response);

        if (array_key_exists('data', $responseData) && array_key_exists('token', $responseData['data'])) {
            $this->token = $responseData['data']['token'];
        }

        $this->client->resetParameters();

        return true;
    }

    /**
     * Send a POST query to the API
     *
     * @param string $endpoint
     * @param mixed  $body
     *
     * @return mixed
     *
     * @throws ApiException
     */
    public function post($endpoint, $body)
    {
        if ($this->token === null) {
            $this->authenticate();

            if ($this->token === null) {
                throw new ApiException(__('Unable to get authentication token for the API'), $this->getLastRequest());
            }
        }

        $this->client->setUri($this->getUriObject($endpoint))
            ->setMethod(Request::METHOD_POST)
            ->setHeaders(['Content-Type' => 'application/json'])
            ->setParameterGet([
                'db'            => $this->chain,
                'store'         => $this->shop,
                'token'         => $this->token,
                'Appsource'     => self::API_APPSOURCE,
                'FastmagApiVer' => self::API_VERSION,
            ]);

        $this->client->setRawBody($this->jsonSerializer->serialize($body));

        $response = $this->send();
        $result = $this->getResponseData($response);
        $this->client->resetParameters();

        return $result;
    }

    /**
     * Send the request, and log it if the configuration is set
     *
     * @throws ApiException
     */
    public function send(): Response
    {
        try {
            $response = $this->client->send();
        } catch (RuntimeException $exception) {
            throw new ApiException(__('Unable to send the request to the API'), '', $exception);
        }

        if ($this->config->isSetFlag(Config::XML_PATH_LOG_API_CALLS)) {
            $logApi = $this->logApiFactory->create();

            $logApi->setEndpoint($this->client->getUri())
                ->setRequest($this->getLastRequest());

            if ($this->client->getResponse() !== null) {
                $logApi->setResponse($this->client->getResponse()->getBody());
            }

            try {
                $this->logApiRepository->save($logApi);
            } catch (CouldNotSaveException $exception) {
                $this->logger->warning(__('Error when trying to log API call: %1', $exception->getMessage())->render());
            }
        }

        return $response;
    }

    /**
     * Get last API request
     *
     * @return string
     */
    public function getLastRequest(): string
    {
        return $this->client->getLastRawRequest();
    }

    /**
     * Get URI object
     *
     * @param string $path
     *
     * @return HttpUri
     *
     * @throws ApiException
     */
    protected function getUriObject($path): HttpUri
    {
        HttpUri::validateHost($this->host);

        try {
            $this->uri->setScheme('https')->setPort(443)->setHost($this->host);
        } catch (InvalidUriPartException $exception) {
            throw new ApiException(__('The endpoint "%1" is invalid', $path), '', $exception);
        }

        $this->uri->setPath($path);

        return $this->uri;
    }

    /**
     * Get response data
     *
     * @param Response $response
     *
     * @return mixed
     *
     * @throws ApiException
     */
    protected function getResponseData($response)
    {
        $responseBody = $response->getBody();
        $responseJson = $this->jsonSerializer->unserialize($responseBody);

        if (array_key_exists('status', $responseJson) && $responseJson['status'] === 'KO') {
            throw new ApiException(
                __($responseJson['Message']),
                $this->getLastRequest(),
                null,
                $responseJson['errorCode']
            );
        }

        return $responseJson;
    }
}
