<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-12
 ******************************************************************************/

namespace Fastmag\Sync\Model\System\Connection;

use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Sql\DirectSqlFactory;
use Fastmag\Sync\Model\System\Connection\Sql\EdiSqlFactory;
use Fastmag\Sync\Model\System\Connection\Sql\SqlInterface;

/**
 * Class Proxy
 *
 * Proxy class hiding the deeper SQL interface class
 */
class Proxy
{
    /** @var string */
    public const CONNECTION_TYPE_DIRECT_SQL = 'direct_sql';

    /** @var string */
    public const CONNECTION_TYPE_EDI_SQL = 'edi_sql';

    /** @var Config $config */
    protected $config;

    /** @var DirectSqlFactory $directSqlFactory */
    protected $directSqlFactory;

    /** @var EdiSqlFactory $ediSqlFactory */
    protected $ediSqlFactory;

    /** @var SqlInterface $connection */
    protected $connection;

    /** @var string $connectionType */
    protected $connectionType;

    /**
     * Proxy constructor.
     *
     * @param Config $config
     * @param DirectSqlFactory $directSqlFactory
     * @param EdiSqlFactory $ediSqlFactory
     */
    public function __construct(
        Config $config,
        DirectSqlFactory $directSqlFactory,
        EdiSqlFactory $ediSqlFactory
    ) {
        $this->config = $config;
        $this->directSqlFactory = $directSqlFactory;
        $this->ediSqlFactory = $ediSqlFactory;
    }

    /**
     * Get the SQL connection, depending on the configuration "native_sql_connection"
     *
     * @return SqlInterface
     *
     * @throws NoConnectionException
     */
    public function getConnection()
    {
        $useNativeSql = $this->config->isSetFlag(Config::XML_PATH_CONNECT_PROXY_NATIVE_SQL_CONNECTION);

        if ($useNativeSql) {
            $this->connection = $this->directSqlFactory->create();
            $this->connection->connect();
            $this->connectionType = self::CONNECTION_TYPE_DIRECT_SQL;
        }

        if (!$useNativeSql || !$this->connection->connect()) {
            $this->connection = $this->ediSqlFactory->create();
            $this->connection->connect();
            $this->connectionType = self::CONNECTION_TYPE_EDI_SQL;
        }

        return $this->connection;
    }

    /**
     * @return string
     */
    public function getConnectionType()
    {
        return $this->connectionType;
    }

    /**
     * @return bool
     */
    public function useConnectionNativeSql()
    {
        return $this->getConnectionType() === self::CONNECTION_TYPE_DIRECT_SQL;
    }
}
