<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-01-21
 ******************************************************************************/

namespace Fastmag\Sync\Process\Entity\ToMagento\Customer;

use Fastmag\Sync\Helper\Data as Helper;
use Fastmag\Sync\Process\Entity\Helper\Attribute\Prefix;
use Fastmag\Sync\Process\EntityInterface;
use Magento\Framework\DataObject;

/**
 * Class Address
 *
 * Entity class for customer address.
 * Save and transform data from Fastmag for Magento easy integration,
 * and transfer it from hydration worker to integration worker
 *
 * @method string|null getAlias()
 * @method int|null getCustomerFastmagId()
 * @method string|null getCompany()
 * @method string|null getPostcode()
 * @method string|null getCity()
 * @method int|null getFastmagOrderId()
 * @method Address setAlias(string $alias)
 * @method Address setCustomerFastmagId(int $customerFastmagId)
 * @method Address setPrefix(string $prefix)
 * @method Address setFirstname(string $firstname)
 * @method Address setLastname(string $lastname)
 * @method Address setCompany(string $company)
 * @method Address setStreetOne(string $streetOne)
 * @method Address setStreetTwo(string $streetTwo)
 * @method Address setPostcode(string $postcode)
 * @method Address setCity(string $city)
 * @method Address setCountryIsoCode(string $countryIsoCode)
 * @method Address setPhoneNumber(string $phoneNumber)
 * @method Address setCellphoneNumber(string $cellphoneNumber)
 * @method Address setFastmagOrderId(int $fastmagOrderId)
 */
class Address extends DataObject implements EntityInterface
{
    /**
     * Get prefix
     *
     * @return string
     */
    public function getPrefix()
    {
        return $this->getData('prefix') !== null
            ? Prefix::getMagentoValue($this->getData('prefix'))
            : Prefix::getDefaultMagentoValue();
    }

    /**
     * Get firstname
     *
     * @return string
     */
    public function getFirstname()
    {
        $firstname = $this->getData('firstname');
        $lastname = $this->getData('lastname');

        if (empty($firstname) && !empty($lastname)) {
            $names = explode(' ', $lastname);
            $firstname = array_pop($names);
        }

        return Helper::capitalize($firstname);
    }

    /**
     * Get lastname
     *
     * @return string
     */
    public function getLastname()
    {
        $firstname = $this->getData('firstname');
        $lastname = $this->getData('lastname');

        if (empty($firstname) && !empty($lastname)) {
            $names = explode(' ', $lastname);
            array_pop($names);
            $lastname = implode(' ', $names);
        }

        return Helper::capitalize($lastname);
    }

    /**
     * Get street
     *
     * @return string[]
     */
    public function getStreet()
    {
        return [$this->getData('street_one'), $this->getData('street_two')];
    }

    /**
     * Get country ISO code
     *
     * @return string
     */
    public function getCountryIsoCode()
    {
        return Helper::upper($this->getData('country_code') ?? 'fr');
    }

    /**
     * Get phone number
     *
     * @return string
     */
    public function getPhoneNumber()
    {
        return Helper::lower($this->getData('phone_number'));
    }

    /**
     * Get cellphone number
     *
     * @return string
     */
    public function getCellphoneNumber()
    {
        return Helper::lower($this->getData('cellphone_number'));
    }

    /**
     * @inheritDoc
     */
    public function export()
    {
        $result = [
            'alias'               => $this->getAlias(),
            'customer_fastmag_id' => $this->getCustomerFastmagId(),
            'prefix'              => $this->getPrefix(),
            'firstname'           => $this->getFirstname(),
            'lastname'            => $this->getLastname(),
            'company'             => $this->getCompany(),
            'street'              => $this->getStreet(),
            'postcode'            => $this->getPostcode(),
            'city'                => $this->getCity(),
            'country_iso_code'    => $this->getCountryIsoCode(),
            'phone_number'        => $this->getPhoneNumber(),
            'cellphone_number'    => $this->getCellphoneNumber(),
            'fastmag_order_id'    => $this->getFastmagOrderId()
        ];

        return array_filter(
            $result,
            static function ($value) {
                return $value !== null && $value !== [];
            }
        );
    }
}
