<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-12-02
 ******************************************************************************/

namespace Fastmag\Sync\Console;

use Exception;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Manager\ToFastmag;
use Fastmag\Sync\Process\Manager\ToMagento;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\Exception\LocalizedException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Class ProcessQueueCommand
 *
 * Command class for jobs queue process
 */
class ProcessQueueCommand extends Command
{
    /** @var string */
    public const INPUT_KEY_SYNC_WAY = 'sync-way';

    /** @var string */
    public const INPUT_KEY_JOB_CODE = 'job-code';

    /** @var string */
    public const INPUT_KEY_JOB_ID = 'job-id';

    /** @var string */
    public const SYNC_WAY_TO_FASTMAG = 'tofastmag';

    /** @var string */
    public const SYNC_WAY_TO_MAGENTO = 'tomagento';

    /** @var string */
    public const SYNC_WAY_BOTH = 'both';

    /** @var Logger $logger */
    protected $logger;

    /** @var Config $config */
    protected $config;

    /** @var ToMagento $toMagento */
    protected $toMagento;

    /** @var ToFastmag $toFastmag */
    protected $toFastmag;

    /** @var State $state */
    protected $state;

    /**
     * ProcessQueueCommand constructor
     *
     * @param Logger      $logger
     * @param Config      $config
     * @param ToMagento   $toMagento
     * @param ToFastmag   $toFastmag
     * @param State       $state
     * @param string|null $name
     */
    public function __construct(
        Logger $logger,
        Config $config,
        ToMagento $toMagento,
        ToFastmag $toFastmag,
        State $state,
        string $name = 'fastmag-sync:jobqueue'
    ) {
        $this->logger = $logger;
        $this->config = $config;
        $this->toMagento = $toMagento;
        $this->toFastmag = $toFastmag;
        $this->state = $state;

        parent::__construct($name);
    }

    /**
     * @inheritDoc
     *
     * @return void
     */
    protected function configure()
    {
        $this->setDescription('Process local jobs queues');

        $values = [self::SYNC_WAY_TO_FASTMAG, self::SYNC_WAY_TO_MAGENTO, self::SYNC_WAY_BOTH];

        $this->setDefinition(
            [
                new InputArgument(
                    self::INPUT_KEY_SYNC_WAY,
                    InputArgument::OPTIONAL,
                    'Sync way ("' . implode('", "', $values) . '")',
                    self::SYNC_WAY_BOTH
                ),
                new InputOption(
                    self::INPUT_KEY_JOB_CODE,
                    '-c',
                    InputOption::VALUE_OPTIONAL,
                    'Job code'
                ),
                new InputOption(
                    self::INPUT_KEY_JOB_ID,
                    '-i',
                    InputOption::VALUE_OPTIONAL,
                    'Job ID (in Magento DB)'
                )
            ]
        );

        parent::configure();
    }

    /**
     * @inheritDoc
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $result = 0;

        try {
            $this->state->setAreaCode(Area::AREA_ADMINHTML);
        } catch (LocalizedException $exception) {
            // Do nothing
        }

        $syncWay = $input->getArgument(self::INPUT_KEY_SYNC_WAY);
        $jobCode = $input->getOption(self::INPUT_KEY_JOB_CODE);
        $jobId = $input->getOption(self::INPUT_KEY_JOB_ID);

        if ($this->isSyncWayValid($syncWay)) {
            $this->logger->debug('Beginning jobs queues processing');

            sleep($this->config->getValue(Config::XML_PATH_JOBQUEUE_ADVANCED_REACTIVITY));

            try {
                if ($syncWay === self::SYNC_WAY_BOTH || $syncWay === self::SYNC_WAY_TO_FASTMAG) {
                    $this->toFastmag->run($jobCode, $jobId);
                }
                if ($syncWay === self::SYNC_WAY_BOTH || $syncWay === self::SYNC_WAY_TO_MAGENTO) {
                    $this->toMagento->run($jobCode, $jobId);
                }
            } catch (Exception $exception) {
                $output->writeln($exception->getMessage() . "\n" . $exception->getTraceAsString());
                $result = 1;
            }
        }

        return $result;
    }

    /**
     * Check if sync way parameter is valid
     *
     * @param string $syncWay
     *
     * @return bool
     */
    protected function isSyncWayValid($syncWay)
    {
        return in_array($syncWay, [self::SYNC_WAY_TO_FASTMAG, self::SYNC_WAY_TO_MAGENTO, self::SYNC_WAY_BOTH], true);
    }
}
