<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2024-01-02
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Process\Entity\ToMagento\Customer as CustomerEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Customer\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Customer\AddressFactory as AddressEntityFactory;
use Fastmag\Sync\Process\Entity\ToMagento\CustomerFactory as CustomerEntityFactory;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Customer
 *
 * Hydration class used for inserting or updating customers from Fastmag to Magento
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Customer extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_customer';

    /** @var CustomerEntityFactory $customerEntityFactory */
    protected $customerEntityFactory;

    /** @var AddressEntityFactory $addressEntityFactory */
    protected $addressEntityFactory;

    /** @var string[] $subordinateWorkersAfter */
    protected $subordinateWorkersAfter = ['tomagento_hydration_customer_address'];

    /**
     * Integration constructor.
     *
     * @param Logger                $logger
     * @param JobRepository         $jobRepository
     * @param Config                $config
     * @param Json                  $jsonSerializer
     * @param Proxy                 $fastmagSql
     * @param CustomerEntityFactory $customerEntityFactory
     * @param AddressEntityFactory  $addressEntityFactory
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        Proxy $fastmagSql,
        CustomerEntityFactory $customerEntityFactory,
        AddressEntityFactory $addressEntityFactory
    ) {
        parent::__construct($logger, $jobRepository, $config, $jsonSerializer, $fastmagSql);

        $this->customerEntityFactory = $customerEntityFactory;
        $this->addressEntityFactory = $addressEntityFactory;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $results = $this->getDataFromFastmag();
        } catch (ProcessException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }

        if (isset($results)) {
            foreach ($this->getJobs()->getItems() as $job) {
                if (array_key_exists($job->getContentId(), $results)) {
                    $this->hydrateJob($job, $results[$job->getContentId()]);
                } else {
                    $exception = new JobException(__('No data found in Fastmag for customer %1', $job->getContentId()));
                    $this->skipJob($job, $exception);
                }

                try {
                    $this->saveJob($job);
                } catch (ProcessException $exception) {
                    $this->invalidateJob($job, $exception);
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $customersIds = $this->jobs->getColumnValues(Job::CONTENT_ID);

        try {
            $sql = 'SELECT Client AS fastmag_id, Actif AS active, Sexe AS gender, Civilite AS prefix,
                    Prenom AS firstname, Nom AS lastname, Email AS email_address, JourNaissance AS birth_day,
                    MoisNaissance AS birth_month, AnneeNaissance AS birth_year, PubParEMail AS subscribe_nl,
                    Remise AS discount_rate, Regroupement AS grouping, DateCreation AS creation_date,
                    ObsClient AS comment, Societe AS company, Adresse1 AS street_one, Adresse2 AS street_two,
                    Adresse3 AS street_three, CodePostal AS postcode, Ville AS city, Telephone AS phone_number,
                    Portable AS cellphone_number,
                    (SELECT CodeIso FROM pays WHERE pays.Pays = clients.Pays) AS country_iso_code
                FROM clients
                WHERE Client IN (' . $this->getFastmagSqlConnection()->escape($customersIds) . ')';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new ProcessException(__(
                'Error when hydrating customers. Message: %1. Customers IDs: %2',
                $exception->getMessage(),
                implode(', ', $customersIds)
            ));
        }

        $result = [];
        foreach ($rows as $row) {
            $customerEntity = $this->generateCustomerEntity($row);
            $addressEntity = $this->generateBillingAddressEntity($row);
            $customerEntity->setBillingAddress($addressEntity);

            $result[$row['fastmag_id']] = $customerEntity;
        }

        return $result;
    }

    /**
     * Generate customer entity with values retrieved by SQL query
     *
     * @param array $row
     *
     * @return CustomerEntity
     */
    protected function generateCustomerEntity($row)
    {
        $customerData = [
            'fastmag_id'    => $row['fastmag_id'],
            'active'        => $row['active'],
            'gender'        => $row['gender'],
            'prefix'        => $row['prefix'],
            'firstname'     => $row['firstname'],
            'lastname'      => $row['lastname'],
            'email_address' => $row['email_address'],
            'birth_day'     => $row['birth_day'],
            'birth_month'   => $row['birth_month'],
            'birth_year'    => $row['birth_year'],
            'subscribe_nl'  => $row['subscribe_nl'],
            'discount_rate' => $row['discount_rate'],
            'grouping'      => $row['grouping'],
            'creation_date' => $row['creation_date'],
            'comment'       => $row['comment']
        ];

        $result = $this->customerEntityFactory->create();
        $result->setData($customerData);

        return $result;
    }

    /**
     * Generate customer billing address entity with values retrieved by SQL query
     *
     * @param array $row
     *
     * @return AddressEntity
     */
    protected function generateBillingAddressEntity($row): AddressEntity
    {
        $billingAddressData = [
            'alias'               => Constants::ATTRIBUTE_ADDRESS_ALIAS_HOME,
            'customer_fastmag_id' => $row['fastmag_id'],
            'prefix'              => $row['prefix'],
            'firstname'           => $row['firstname'],
            'lastname'            => $row['lastname'],
            'company'             => $row['company'],
            'street_one'          => $row['street_one'],
            'street_two'          => $row['street_two'],
            'street_three'        => $row['street_three'],
            'postcode'            => $row['postcode'],
            'city'                => $row['city'],
            'country_iso_code'    => $row['country_iso_code'],
            'phone_number'        => $row['phone_number'],
            'cellphone_number'    => $row['cellphone_number']
        ];

        $result = $this->addressEntityFactory->create();
        $result->setData($billingAddressData);

        return $result;
    }
}
