<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-09-26
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\Order as SyncedOrder;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration as IntegrationTrait;
use Magento\Framework\Serialize\Serializer\Json;
use stdClass;

abstract class Transfer extends Worker
{
    use IntegrationTrait;

    /** @var Config $config */
    protected $config;

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /** @var string $transactionType */
    protected $transactionType = OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_TRANSFER;

    /** @var Job $currentJob */
    protected $currentJob;

    /**
     * Transfer constructor
     *
     * @param Logger        $logger
     * @param JobRepository $jobRepository
     * @param Api           $api
     * @param Config        $config
     * @param Json          $jsonSerializer
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Api $api,
        Config $config,
        Json $jsonSerializer
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->api = $api;
        $this->config = $config;
        $this->jsonSerializer = $jsonSerializer;
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_ORDER_EXPORT_ENABLE);
    }

    /**
     * Save transfer synchronizations
     *
     * @param stdClass $request
     * @param array    $response
     * @param string   $requestAt
     * @param string   $endpoint
     * @param string   $originShop
     * @param string   $targetShop
     * @param string   $type
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveSyncedTransfer(
        $request,
        $response,
        $requestAt,
        $endpoint,
        $originShop,
        $targetShop,
        $type = Constants::TRANSFER_TYPE_ORDER
    ) {
        /** @var SyncedOrder $syncedTransfer */
        $syncedTransfer = $this->syncedOrderFactory->create();

        $syncedTransfer->setOrderId($this->currentJob->getEntity()->getMagentoId())
            ->setTransactionId($response['Id'])
            ->setTransactionType($this->getTransferTransactionType())
            ->setRequest($this->jsonSerializer->serialize($request))
            ->setEndpoint($endpoint)
            ->setStatus($response['status'])
            ->setRequestAt($requestAt)
            ->setResultAt(date('Y-m-d H:i:s'))
            ->setOriginShop($originShop)
            ->setTargetShop($targetShop);

        try {
            $syncedTransfer = $this->syncedOrderRepository->save($syncedTransfer);

            /** @var OrderEntity $orderEntity */
            $orderEntity = $this->currentJob->getEntity();
            $orderEntity->addTransfer($syncedTransfer, $type);
        } catch (Exception $exception) {
            throw new JobException(__(
                'Unable to save transfer for order #%1 synchronization on Magento DB',
                $this->currentJob->getEntity()->getMagentoId()
            ));
        }
    }

    /**
     * Get transfer transaction type
     *
     * @return string
     */
    protected function getTransferTransactionType()
    {
        return $this->transactionType;
    }
}
