<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2023-12-28
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Customer;

use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Process\Entity\ToMagento\Customer as CustomerEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Customer\Address as AddressEntity;
use Fastmag\Sync\Process\Worker\ToMagento\Integration\Address as DefaultAddress;
use Magento\Customer\Api\AddressRepositoryInterface as AddressRepository;
use Magento\Customer\Api\Data\AddressInterface as AddressModel;
use Magento\Customer\Api\Data\AddressInterfaceFactory as AddressFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\LocalizedException;

/**
 * Class Address
 *
 * Integration worker for tomagento_integration_customer_create and tomagento_integration_customer_update jobs
 * Deals with customer address creation and update
 */
class Address extends DefaultAddress
{
    /** @inheritDoc */
    protected $code = 'tomagento_integration_customer_address';

    /**
     * Address constructor
     *
     * @param Logger                $logger
     * @param AddressRepository     $addressRepository
     * @param AddressFactory        $addressFactory
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param Config                $config
     * @param JobRepository         $jobRepository
     */
    public function __construct(
        Logger $logger,
        AddressRepository $addressRepository,
        AddressFactory $addressFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Config $config,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger, $addressRepository, $addressFactory, $searchCriteriaBuilder);

        $this->jobRepository = $jobRepository;
        $this->config = $config;
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_IMPORT_ENABLE);
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var CustomerEntity $customerEntity */
        $customerEntity = $job->getEntity();

        $this->saveAddress($customerEntity);
        $this->saveAddress($customerEntity, 'shipping');
    }

    /**
     * Save address of the customer
     *
     * @param CustomerEntity $customerEntity
     * @param string         $type
     *
     * @throws JobException
     */
    protected function saveAddress($customerEntity, $type = 'billing')
    {
        $addressEntity = $customerEntity->getBillingAddress();
        if ($type === 'shipping') {
            $addressEntity = $customerEntity->getShippingAddress();
        }

        if ($addressEntity !== null && $this->isAddressValid($addressEntity)) {
            $address = $this->checkIfAddressExists($customerEntity->getMagentoId(), $addressEntity->getAlias());
            if (!$address) {
                $address = $this->addressFactory->create();
                $address->setCustomerId($customerEntity->getMagentoId());
            }

            if ($type === 'billing' && !$customerEntity->getMagentoDefaultShipping()) {
                $address->setIsDefaultBilling(true);
            }
            if ($type === 'shipping' && !$customerEntity->getMagentoDefaultShipping()) {
                $address->setIsDefaultShipping(true);
            }

            $this->saveEntity($address, $addressEntity);
        }
    }

    /**
     * Save address with entity attributes
     *
     * @param AddressModel  $address
     * @param AddressEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveEntity($address, $entity)
    {
        $this->setAddressData($address, $entity);

        try {
            $this->addressRepository->save($address);
        } catch (LocalizedException $exception) {
            throw new JobException(__('Unable to sync the address: %1', $exception->getMessage()));
        }
    }

    /**
     * Check if address has street, postcode, city and alias filled at least
     *
     * @param AddressEntity $address
     *
     * @return bool
     *
     * @throws JobException
     */
    protected function isAddressValid($address)
    {
        if ($address->getStreet() === null) {
            throw new JobException(__('The address given does not have the street fields filled'));
        }

        if ($address->getPostcode() === null) {
            throw new JobException(__('The address given does not have post code field filled'));
        }

        if ($address->getCity() === null) {
            throw new JobException(__('The address given does not have the city field filled'));
        }

        if ($address->getCountryIsoCode() === null) {
            throw new JobException(__('The address given does not have the country field filled'));
        }

        if ($address->getPhoneNumber() === null && $address->getCellphoneNumber() === null) {
            throw new JobException(__('The address given does not have a phone number filled'));
        }

        if ($address->getAlias() === null) {
            throw new JobException(__('The address given does not have an alias'));
        }

        return true;
    }

    /**
     * Check if an address in Magento DB has the alias for the customer ID given in param and returns it if it exists
     *
     * @param int    $customerId
     * @param string $addressAlias
     *
     * @return AddressModel|false
     *
     * @throws JobException
     */
    protected function checkIfAddressExists($customerId, $addressAlias)
    {
        try {
            $this->searchCriteriaBuilder->addFilter('parent_id', $customerId)
                ->addFilter(Constants::ATTRIBUTE_ADDRESS_ALIAS_CODE, $addressAlias);
            $searchCriteria = $this->searchCriteriaBuilder->create();

            $addresses = $this->addressRepository->getList($searchCriteria);
        } catch (LocalizedException $exception) {
            return false;
        }

        if ($addresses->getTotalCount() > 1) {
            throw new JobException(__(
                'It seems there is more than one address matching the alias "%1". Please review them ASAP',
                $addressAlias
            ));
        }

        if ($addresses->getTotalCount() === 0) {
            $address = false;
        } else {
            $address = $addresses->getItems()[0];
        }

        return $address;
    }

    /**
     * Set entity data into address model
     *
     * @param AddressModel  $address
     * @param AddressEntity $entity
     *
     * @return void
     */
    protected function setAddressData($address, $entity)
    {
        // TODO: add scope to get real config value if there's an override on a specific website/store
        $addressLinesLimit = $this->config->getValue('customer/address/street_lines');

        $address->setPrefix($entity->getPrefix())
            ->setFirstname($entity->getFirstname())
            ->setLastname($entity->getLastname())
            ->setCompany($entity->getCompany())
            ->setStreet(array_slice($entity->getStreet(), 0, $addressLinesLimit))
            ->setPostcode($entity->getPostcode())
            ->setCity($entity->getCity())
            ->setCountryId($entity->getCountryIsoCode())
            ->setTelephone(
                $entity->getCellphoneNumber() === '' ? $entity->getPhoneNumber() : $entity->getCellphoneNumber()
            );

        $address->setCustomAttribute(Constants::ATTRIBUTE_ADDRESS_ALIAS_CODE, $entity->getAlias());
    }
}
