<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2022 HOMEMADE.IO SAS
 * @date      2022-03-31
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Exception\NoBillingAddressException;
use Fastmag\Sync\Process\Entity\ToFastmag\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Address as AbstractAddress;

/**
 * Class Address
 *
 * Hydration class used for inserting or updating customers addresses from Magento to Fastmag
 */
class Address extends AbstractAddress
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_customer_address';

    /**
     * @inheritDoc
     */
    public function run()
    {
        $results = $this->getDataFromMagento();

        if (is_array($results)) {
            foreach ($this->getJobs()->getItems() as $job) {
                if (!$job->isInError()) {
                    $customerId = $job->getContentId();

                    if (!array_key_exists($customerId, $results) || $results[$customerId]['billing_address'] === null) {
                        $exception = new NoBillingAddressException(__(
                            'Unable to create customer #%1 on Fastmag: no billing address found.',
                            $customerId
                        ));
                        $this->skipJob($job, $exception);
                    } else {
                        /** @var CustomerEntity $customerEntity */
                        $customerEntity = $job->getEntity();
                        $customerEntity->setBillingAddress($results[$customerId]['billing_address'])
                            ->setShippingAddress($results[$customerId]['shipping_address']);

                        $this->hydrateJob($job, $customerEntity);
                        $this->debug($job, 'Customer\'s addresses hydrated');
                    }

                    $this->saveJob($job);
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @return AddressEntity[]
     */
    protected function getDataFromMagento()
    {
        $customersIds = $this->getJobs()->getColumnValues(Job::CONTENT_ID);
        $result = [];

        foreach ($customersIds as $customerId) {
            $billingAddress = $this->getAddressEntity($customerId);
            $shippingAddress = $this->getAddressEntity($customerId, 'shipping');

            $hydratedData = [
                'billing_address'  => $billingAddress,
                'shipping_address' => $shippingAddress
            ];

            $result[$customerId] = $hydratedData;
        }

        return $result;
    }

    /**
     * Get customer's default addresses
     *
     * @param int    $customerId
     * @param string $type
     *
     * @return AddressEntity|null
     */
    protected function getAddressEntity($customerId, $type = 'billing')
    {
        $result = null;
        $addressObject = null;

        try {
            $customer = $this->customerRepository->getById($customerId);
        } catch (Exception $exception) {
            $customer = null;
        }

        if ($customer !== null) {
            if ($type === 'billing') {
                $addressId = $customer->getDefaultBilling();
            } else {
                $addressId = $customer->getDefaultShipping();
            }

            if ($addressId) {
                try {
                    $addressObject = $this->addressRepository->getById($addressId);
                } catch (Exception $exception) {
                    // Do nothing
                }
            }
        }

        if ($addressObject !== null) {
            $result = $this->generateAddressEntity(
                $addressObject,
                $this->getAddressAlias($addressObject)
            );
        }

        return $result;
    }
}
