<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2024-01-12
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Customer;

use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Customer;
use Magento\Store\Model\ScopeInterface;

/**
 * Class Deletion
 *
 * Integration class used for deleting customers from Magento to Fastmag
 */
class Deletion extends Customer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_customer_deletion';

    /** @var string $hydrationWorker */
    protected $hydrationWorker = 'tofastmag_hydration_customer_deletion';

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->getValue(Config::XML_PATH_CUSTOMER_GDPR_DELETE_DATA_ACTION)
            !== Constants::CUSTOMER_GDPR_ACTION_NOTHING;
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var CustomerEntity $entity */
        $entity = $job->getEntity();

        if (!$entity->getFastmagId()) {
            $fastmagId = $this->getFastmagIdByEmail($entity->getEmailAddress());

            if ($fastmagId) {
                $entity->setFastmagId($fastmagId);

                $this->saveCustomerSync($entity->getMagentoId(), $fastmagId);
            }
        }

        if ($entity->getFastmagId()) {
            $this->sendInstructionToFastmag($entity);
            $this->saveCustomerSync($entity->getMagentoId(), $entity->getFastmagId(), true);
        }
    }

    /**
     * Send deletion action to Fastmag
     *
     * @param CustomerEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendInstructionToFastmag($entity)
    {
        $action = $this->config->getValue(Config::XML_PATH_CUSTOMER_GDPR_DELETE_DATA_ACTION);

        if ($action === Constants::CUSTOMER_GDPR_ACTION_NOTICE) {
            $this->sendDeletionNotice($entity);
        } elseif ($action === Constants::CUSTOMER_GDPR_ACTION_ANONYMIZE) {
            $this->sendAnonymizationRequest($entity);
        } elseif ($action === Constants::CUSTOMER_GDPR_ACTION_DELETE) {
            $this->sendDeletionRequest($entity);
        }
    }

    /**
     * Send CONSIGNE for the customer deletion to Fastmag
     *
     * @param CustomerEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendDeletionNotice($entity)
    {
        $fastmagShop = $this->config->getValue(
            Config::XML_PATH_ORDER_WORKFLOW_SHOP,
            ScopeInterface::SCOPE_STORES,
            $entity->getStoreId()
        );

        $data = (object)['Consignes' => [(object)[
            'Destinataire' => $fastmagShop,
            'Sujet'        => 'SUPPRESSION COMPTE CLIENT',
            'Message'      => 'Bonjour,{CR}{LF}Le client ayant l\'ID Fastmag #' . $entity->getFastmagId()
                . ' a demande a etre supprime dans Fastmag, conformement au RGPD.',
            'Expediteur'   => $this->config->getValue(Config::XML_PATH_CONNECT_API_SHOP),
            'Slide'        => '1'
        ]]];

        try {
            $response = $this->api->post('/boa/consigne/create/index.ips', $data);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException(__($message), $this->api->getLastRequest());
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send instruction for the deletion of the customer #%1 on Fastmag through API. Code: %2. Message: %3. Cause: %4. Data sent: %5',
                $entity->getMagentoId(),
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                $exception->getRequest()
            ));
        }
    }

    /**
     * Send anonymization request to Fastmag
     *
     * @param CustomerEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendAnonymizationRequest(CustomerEntity $entity): void
    {
        $data = (object)[
            'Email'  => $entity->getEmailAddress(true),
            'Client' => $entity->getFastmagId()
        ];

        try {
            $response = $this->api->post('/boa/client/anonymize/index.ips', $data);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException(__($message), $this->api->getLastRequest());
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to anonymize customer #%1 data on Fastmag through API. Code: %2. Message: %3. Source: %4. Data sent: %5',
                $entity->getMagentoId(),
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                $exception->getRequest()
            ));
        }
    }

    /**
     * Send deletion request to Fastmag
     *
     * @param CustomerEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendDeletionRequest(CustomerEntity $entity): void
    {
        $data = (object)[
            'Email'  => $entity->getEmailAddress(true),
            'Client' => $entity->getFastmagId()
        ];

        try {
            $response = $this->api->post('/boa/client/delete/index.ips', $data);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException(__($message), $this->api->getLastRequest());
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to delete customer #%1 data on Fastmag through API. Code: %2. Message: %3. Source: %4. Data sent: %5',
                $entity->getMagentoId(),
                $exception->getCode(),
                $exception->getMessage(),
                ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : ''),
                $exception->getRequest()
            ));
        }
    }
}
