<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2024 HOMEMADE.IO SAS
 * @date      2024-01-02
 ******************************************************************************/

namespace Fastmag\Sync\Process\Entity\ToFastmag;

use Fastmag\Sync\Process\Entity\Helper\Attribute\Prefix;
use Fastmag\Sync\Process\EntityInterface;
use Magento\Framework\DataObject;

/**
 * Class Address
 *
 * Entity class for address.
 * Save and transform data from Magento for Fastmag easy integration,
 * and transfer it from hydration worker to integration worker
 *
 * @method string|null getAlias()
 * @method int|null getMagentoId()
 * @method string|null getFirstname()
 * @method string|null getLastname()
 * @method string|null getCompany()
 * @method string[]|null getStreet()
 * @method string|null getPostcode()
 * @method string|null getCity()
 * @method string|null getCountryId()
 * @method string|null getPhoneNumber()
 * @method Address setAlias(string $alias)
 * @method Address setMagentoId(int $magentoId)
 * @method Address setPrefix(string $prefix)
 * @method Address setFirstname(string $firstname)
 * @method Address setLastname(string $lastname)
 * @method Address setCompany(string $company)
 * @method Address setStreet(string[] $street)
 * @method Address setPostcode(string $postcode)
 * @method Address setCity(string $city)
 * @method Address setCountryId(string $countryId)
 * @method Address setPhoneNumber(string $phoneNumber)
 * @method Address setArchive(bool $archive)
 * @method Address setIsDefault(bool $isDefault)
 */
class Address extends DataObject implements EntityInterface
{
    /**
     * Get prefix
     *
     * @return string|null
     */
    public function getPrefix(): ?string
    {
        return $this->getData('prefix') !== null ? Prefix::getFastmagValue($this->getData('prefix')) : null;
    }

    /**
     * Get first line of street
     *
     * @return string|null
     */
    public function getStreetOne(): ?string
    {
        return is_array($this->getData('street')) ? $this->getData('street')[0] : null;
    }

    /**
     * Get second line of street
     *
     * @return string|null
     */
    public function getStreetTwo(): ?string
    {
        return (
            is_array($this->getData('street')) && count($this->getData('street')) > 1
            ? $this->getData('street')[1]
            : null
        );
    }

    /**
     * Get third line of street
     *
     * @return string|null
     */
    public function getStreetThree(): ?string
    {
        return (
        is_array($this->getData('street')) && count($this->getData('street')) > 2
            ? $this->getData('street')[2]
            : null
        );
    }

    /**
     * Get is archived token
     *
     * @return bool|null
     */
    public function getArchive(): ?bool
    {
        return $this->getData('archive') !== null ? (bool)$this->getData('archive') : null;
    }

    /**
     * Get is default token
     *
     * @return bool|null
     */
    public function getIsDefault(): ?bool
    {
        return $this->getData('is_default') !== null ? (bool)$this->getData('is_default') : null;
    }

    /**
     * @inheritDoc
     */
    public function export(): array
    {
        $result = [
            'alias'        => $this->getAlias(),
            'magento_id'   => $this->getMagentoId(),
            'prefix'       => $this->getPrefix(),
            'lastname'     => $this->getLastname(),
            'firstname'    => $this->getFirstname(),
            'company'      => $this->getCompany(),
            'street'       => [$this->getStreetOne(), $this->getStreetTwo(), $this->getStreetThree()],
            'postcode'     => $this->getPostcode(),
            'city'         => $this->getCity(),
            'country_id'   => $this->getCountryId(),
            'phone_number' => $this->getPhoneNumber(),
            'archive'      => $this->getArchive(),
            'is_default'   => $this->getIsDefault()
        ];

        return array_filter(
            $result,
            static function ($value) {
                return $value !== null && $value !== [];
            }
        );
    }
}
