<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2023 HOMEMADE.IO SAS
 * @date      2023-05-23
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Order;

use Exception;
use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Data\CustomerInterface as SyncedCustomerInterface;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\CustomerFactory as CustomerEntityFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer as AbstractCustomer;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Newsletter\Model\Subscriber;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\OrderRepositoryInterface as OrderRepository;

/**
 * Class Customer
 *
 * Hydration class used for inserting or updating orders customers from Magento to Fastmag
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Customer extends AbstractCustomer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_order_customer';

    /** @var OrderRepository $orderRepository */
    protected OrderRepository $orderRepository;

    /** @var array $jobsWithCustomers */
    protected array $jobsWithCustomers = [];

    /** @var array $ordersPassedAsGuest */
    protected array $ordersPassedAsGuest = [];

    /**
     * Customer constructor.
     *
     * @param Logger                   $logger
     * @param JobRepository            $jobRepository
     * @param Config                   $config
     * @param Json                     $jsonSerializer
     * @param SyncedCustomerRepository $syncedCustomerRepository
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param CustomerRepository       $customerRepository
     * @param CustomerEntityFactory    $customerEntityFactory
     * @param Subscriber               $subscriber
     * @param OrderRepository          $orderRepository
     *
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        Logger                   $logger,
        JobRepository            $jobRepository,
        Config                   $config,
        Json                     $jsonSerializer,
        SyncedCustomerRepository $syncedCustomerRepository,
        SearchCriteriaBuilder    $searchCriteriaBuilder,
        CustomerRepository       $customerRepository,
        CustomerEntityFactory    $customerEntityFactory,
        Subscriber               $subscriber,
        OrderRepository          $orderRepository
    ) {
        parent::__construct(
            $logger,
            $jobRepository,
            $config,
            $jsonSerializer,
            $syncedCustomerRepository,
            $searchCriteriaBuilder,
            $customerRepository,
            $customerEntityFactory,
            $subscriber
        );

        $this->orderRepository = $orderRepository;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->getDataFromMagento();
        } catch (Exception $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
                $this->saveJob($job);
            }
        }

        if (is_array($this->entities)) {
            foreach ($this->getJobs()->getItems() as $job) {
                /** @var OrderEntity $orderEntity */
                $orderEntity = $job->getEntity();

                if (isset($this->entities[$job->getContentId()])) {
                    $orderEntity->setCustomer($this->entities[$job->getContentId()]);

                    $this->hydrateJob($job, $orderEntity);
                    $this->debug($job, 'Order\'s customer hydrated');
                } else {
                    $exception = new JobException(__('No matching customer data found for job'));
                    $this->skipJob($job, $exception);
                }

                $this->saveJob($job);
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @return CustomerEntity[]
     *
     * @throws ProcessException
     */
    protected function getDataFromMagento()
    {
        $this->getCustomerData();

        if (count($this->jobsWithCustomers) > 0) {
            foreach ($this->jobsWithCustomers as $jobId => $customerId) {
                $customerEntity = $this->customerEntityFactory->create();
                $customerEntity->setMagentoId($customerId);
                $this->entities[$jobId] = $customerEntity;
            }

            $customerIds = array_values($this->jobsWithCustomers);

            $this->getSyncedCustomers($customerIds);
            $this->getMagentoCustomersData($customerIds);
        }

        if (count($this->ordersPassedAsGuest) > 0) {
            $this->generateFakeCustomerEntities();
        }

        return $this->entities;
    }

    /**
     * Retrieve the customer IDs for the orders' jobs
     *
     * @return void
     */
    protected function getCustomerData()
    {
        foreach ($this->getJobs()->getItems() as $job) {
            /** @var OrderEntity $orderEntity */
            $orderEntity = $job->getEntity();

            $transactionType = $orderEntity->getTransactionType();
            $orderId = $orderEntity->getMagentoId();

            if ($orderId !== null && $this->useCustomerData($transactionType)) {
                if ($orderEntity->getIsPassedAsGuest()) {
                    if (!array_key_exists($orderId, $this->ordersPassedAsGuest)) {
                        $this->ordersPassedAsGuest[$orderId] = [];
                    }

                    $this->ordersPassedAsGuest[$orderId][] = $job->getContentId();
                } else {
                    $customerId = $orderEntity->getCustomerId();
                    if ($customerId === null) {
                        continue;
                    }

                    $this->jobsWithCustomers[$job->getContentId()] = $customerId;
                }
            }
        }
    }

    /**
     * Override to get the job with the first associated order of the customer
     *
     * @inheritDoc
     */
    protected function getSyncedCustomers($customerIds)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(SyncedCustomerInterface::MAGENTO_CUSTOMER_ID, $customerIds, 'in')
            ->create();

        $syncedCustomerList = $this->syncedCustomerRepository->getList($searchCriteria);

        foreach ($syncedCustomerList->getItems() as $syncedCustomer) {
            $job = $this->getJob(
                array_search(
                    $syncedCustomer->getMagentoCustomerId(),
                    $this->jobsWithCustomers,
                    true
                )
            );

            if ($job === null) {
                throw new JobException(
                    __('There\'s no order job for customer ID #%1', $syncedCustomer->getMagentoCustomerId())
                );
            }

            $this->debug($job, 'Customer already synced, Fastmag ID: ' . $syncedCustomer->getFastmagCustomerId());

            $currentEntity = $this->getCustomerEntity($syncedCustomer->getMagentoCustomerId());
            $currentEntity->setFastmagId($syncedCustomer->getFastmagCustomerId());
        }
    }

    /**
     * @inheritDoc
     */
    protected function getCustomerEntity($customerId)
    {
        return $this->entities[array_search($customerId, $this->jobsWithCustomers, true)];
    }

    /**
     * Returns true if the transaction use customer data
     *
     * @param string|null $transactionType
     *
     * @return bool
     */
    protected function useCustomerData($transactionType)
    {
        return $transactionType === null || in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_ORDER,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE
            ],
            true
        );
    }

    /**
     * Generate customer entities with the data of the orders
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function generateFakeCustomerEntities()
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(OrderInterface::ENTITY_ID, array_keys($this->ordersPassedAsGuest), 'in')
            ->create();

        $ordersList = $this->orderRepository->getList($searchCriteria);

        try {
            foreach ($ordersList->getItems() as $order) {
                $customerEntity = $this->customerEntityFactory->create();

                $billingAddress = $order->getBillingAddress();
                if ($billingAddress !== null) {
                    $customerEntity->setPrefix($billingAddress->getPrefix())
                        ->setLastname($billingAddress->getLastname())
                        ->setFirstname($billingAddress->getFirstname())
                        ->setEmailAddress($order->getCustomerEmail())
                        ->setStoreId($order->getStoreId())
                        ->setSubscribeNewsletter(false);
                }

                foreach ($this->ordersPassedAsGuest[$order->getEntityId()] as $jobId) {
                    $this->entities[$jobId] = $customerEntity;
                }
            }
        } catch (Exception $exception) {
            throw new ProcessException(__(
                'Error when hydrating orders passed as guest. Message: %1. Orders IDs: %2',
                $exception->getMessage(),
                implode(', ', array_keys($this->ordersPassedAsGuest))
            ));
        }
    }
}
